<?php
/**
 * Authentication Helper Functions
 */

// Suppress errors in API responses
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * Check if user is logged in
 */
function isLoggedIn()
{
    return isset($_SESSION['user_id']) && isset($_SESSION['role']);
}

/**
 * Check if user is admin
 */
function isAdmin()
{
    return isLoggedIn() && $_SESSION['role'] === 'admin';
}

/**
 * Check if user is regular user
 */
function isUser()
{
    return isLoggedIn() && $_SESSION['role'] === 'user';
}

/**
 * Require login - redirect to login if not authenticated
 */
function requireLogin()
{
    if (!isLoggedIn()) {
        header('Location: /app/login.php');
        exit;
    }

    // Check if maintenance mode is ON and user is not admin
    if (!isAdmin()) {
        $conn = getDBConnection();
        $maintenanceSql = "SELECT setting_value FROM settings WHERE setting_key = 'maintenance_mode'";
        $maintenanceResult = $conn->query($maintenanceSql);

        if ($maintenanceResult && $maintenanceResult->num_rows > 0) {
            $maintenanceMode = (int) $maintenanceResult->fetch_assoc()['setting_value'] === 1;

            if ($maintenanceMode) {
                // Get user name before destroying session
                $userId = (int) $_SESSION['user_id'];
                $userSql = "SELECT name FROM users WHERE id = $userId";
                $userResult = $conn->query($userSql);
                $userName = 'User';
                if ($userResult && $userResult->num_rows > 0) {
                    $userName = $userResult->fetch_assoc()['name'];
                }

                $conn->close();
                session_destroy();
                header('Location: /app/login.php?maintenance=1&name=' . urlencode($userName));
                exit;
            }
        }
        $conn->close();
    }
}

/**
 * Require admin - redirect if not admin
 */
function requireAdmin()
{
    requireLogin();
    if (!isAdmin()) {
        header('Location: /app/user/dashboard.php');
        exit;
    }
}

/**
 * Require user - redirect if not user
 */
function requireUser()
{
    requireLogin();
    if (!isUser()) {
        header('Location: /app/admin/dashboard.php');
        exit;
    }
}

/**
 * Login user
 */
function loginUser($email, $password)
{
    $conn = getDBConnection();

    // Check maintenance mode first
    $maintenanceSql = "SELECT setting_value FROM settings WHERE setting_key = 'maintenance_mode'";
    $maintenanceResult = $conn->query($maintenanceSql);
    $maintenanceMode = false;

    if ($maintenanceResult && $maintenanceResult->num_rows > 0) {
        $maintenanceMode = (int) $maintenanceResult->fetch_assoc()['setting_value'] === 1;
    }

    $email = $conn->real_escape_string($email);
    $sql = "SELECT * FROM users WHERE email = '$email'";
    $result = $conn->query($sql);

    if ($result && $result->num_rows > 0) {
        $user = $result->fetch_assoc();

        // Check if maintenance mode is ON and user is not admin
        if ($maintenanceMode && $user['role'] !== 'admin') {
            $userName = $user['name'];
            $conn->close();
            return ['success' => false, 'message' => "Sorry $userName, System is under maintenance. Wait sometime."];
        }

        // Check if user is blocked
        if ($user['status'] === 'blocked') {
            $conn->close();
            return ['success' => false, 'message' => 'Your account has been blocked. Please contact support.'];
        }

        // Verify password
        if (password_verify($password, $user['password'])) {
            // Set session variables
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['name'] = $user['name'];
            $_SESSION['email'] = $user['email'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['daily_limit'] = $user['daily_limit'];

            // Log the login (silently fail if table doesn't exist yet)
            try {
                $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
                $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
                $ipAddress = $conn->real_escape_string($ipAddress);
                $userAgent = $conn->real_escape_string($userAgent);

                $logSql = "INSERT INTO login_logs (user_id, ip_address, user_agent) 
                           VALUES ({$user['id']}, '$ipAddress', '$userAgent')";
                @$conn->query($logSql); // @ suppresses error if table doesn't exist

                // Update last login time
                $updateLoginSql = "UPDATE users SET last_login = NOW() WHERE id = {$user['id']}";
                $conn->query($updateLoginSql);
            } catch (Exception $e) {
                // Silently ignore if login_logs table doesn't exist
            }

            $conn->close();
            return [
                'success' => true,
                'role' => $user['role'],
                'redirect' => $user['role'] === 'admin' ? '/app/admin/dashboard.php' : '/app/user/dashboard.php'
            ];
        }
    }

    $conn->close();
    return ['success' => false, 'message' => 'Invalid email or password'];
}

/**
 * Register new user
 */
function registerUser($name, $email, $mobile, $password, $referralCode = null)
{
    $conn = getDBConnection();

    // Check if email already exists
    $email = $conn->real_escape_string($email);
    $checkSql = "SELECT id FROM users WHERE email = '$email'";
    $result = $conn->query($checkSql);

    if ($result->num_rows > 0) {
        $conn->close();
        return ['success' => false, 'message' => 'Email already registered'];
    }

    // Get default daily limit from settings
    $settingSql = "SELECT setting_value FROM settings WHERE setting_key = 'default_daily_limit'";
    $settingResult = $conn->query($settingSql);
    $defaultLimit = 10;
    if ($settingResult && $settingResult->num_rows > 0) {
        $defaultLimit = (int) $settingResult->fetch_assoc()['setting_value'];
    }

    // Get referral settings
    $referralEnabled = false;
    $refereeReward = 10;
    $referrerReward = 10;

    $refSettingSql = "SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('referral_enabled', 'referee_reward', 'referrer_reward')";
    $refSettingResult = $conn->query($refSettingSql);
    if ($refSettingResult) {
        while ($row = $refSettingResult->fetch_assoc()) {
            if ($row['setting_key'] == 'referral_enabled') {
                $referralEnabled = (int) $row['setting_value'] === 1;
            } elseif ($row['setting_key'] == 'referee_reward') {
                $refereeReward = (int) $row['setting_value'];
            } elseif ($row['setting_key'] == 'referrer_reward') {
                $referrerReward = (int) $row['setting_value'];
            }
        }
    }

    // Validate referral code if provided
    $referrerId = null;
    $bonusSearches = 0;

    if ($referralEnabled && $referralCode) {
        $referralCode = $conn->real_escape_string($referralCode);
        $refCheckSql = "SELECT id, name FROM users WHERE referral_code = '$referralCode'";
        $refCheckResult = $conn->query($refCheckSql);

        if ($refCheckResult && $refCheckResult->num_rows > 0) {
            $referrer = $refCheckResult->fetch_assoc();
            $referrerId = $referrer['id'];
            $bonusSearches = $refereeReward; // New user gets bonus
        }
    }

    // Hash password
    $hashedPassword = password_hash($password, PASSWORD_BCRYPT);

    // Generate unique referral code for new user
    $newUserReferralCode = generateUniqueReferralCode($name, $conn);

    // Insert user
    $name = $conn->real_escape_string($name);
    $mobile = $conn->real_escape_string($mobile);
    $referredBy = $referralCode ? "'$referralCode'" : 'NULL';

    // Set daily limit based on referral
    $userDailyLimit = $referrerId ? $refereeReward : $defaultLimit;

    $sql = "INSERT INTO users (name, email, mobile, password, role, status, daily_limit, referral_code, referred_by, referral_bonus) 
            VALUES ('$name', '$email', '$mobile', '$hashedPassword', 'user', 'active', $userDailyLimit, '$newUserReferralCode', $referredBy, 0)";

    if ($conn->query($sql)) {
        $userId = $conn->insert_id;

        // Create default nav permissions
        $permSql = "INSERT INTO user_nav_permissions (user_id, dashboard, search, history, profile, support) 
                    VALUES ($userId, 1, 1, 1, 1, 1)";
        $conn->query($permSql);

        // If referred by someone, create referral record and give reward to referrer
        if ($referrerId) {
            // Create referral record
            $refRecordSql = "INSERT INTO referrals (referrer_id, referee_id, referral_code, reward_given, status) 
                            VALUES ($referrerId, $userId, '$referralCode', $referrerReward, 'completed')";
            $conn->query($refRecordSql);

            // Give bonus to referrer (both daily_limit and referral_bonus)
            $updateReferrerSql = "UPDATE users SET 
                                    daily_limit = daily_limit + $referrerReward,
                                    referral_bonus = referral_bonus + $referrerReward 
                                  WHERE id = $referrerId";
            $conn->query($updateReferrerSql);
        }

        $conn->close();

        $message = 'Registration successful!';
        if ($referrerId) {
            $message .= " You got $refereeReward searches!";
        } else {
            $message .= " You got $defaultLimit searches!";
        }

        return ['success' => true, 'message' => $message];
    }

    $conn->close();
    return ['success' => false, 'message' => 'Registration failed. Please try again.'];
}

/**
 * Generate unique referral code
 */
function generateUniqueReferralCode($name, $conn)
{
    do {
        $namePrefix = strtoupper(substr(preg_replace('/[^a-zA-Z]/', '', $name), 0, 3));
        if (strlen($namePrefix) < 3) {
            $namePrefix = str_pad($namePrefix, 3, 'X');
        }

        $code = $namePrefix . rand(1000, 9999);

        // Check if code already exists
        $checkSql = "SELECT id FROM users WHERE referral_code = '$code'";
        $result = $conn->query($checkSql);
    } while ($result && $result->num_rows > 0);

    return $code;
}

/**
 * Logout user
 */
function logoutUser()
{
    session_destroy();
    header('Location: /app/login.php');
    exit;
}

/**
 * Get current user data
 */
function getCurrentUser()
{
    if (!isLoggedIn()) {
        return null;
    }

    $conn = getDBConnection();
    $userId = (int) $_SESSION['user_id'];

    $sql = "SELECT * FROM users WHERE id = $userId";
    $result = $conn->query($sql);

    $user = null;
    if ($result && $result->num_rows > 0) {
        $user = $result->fetch_assoc();
    }

    $conn->close();
    return $user;
}

/**
 * Get user nav permissions
 */
function getUserNavPermissions($userId = null)
{
    if ($userId === null) {
        $userId = $_SESSION['user_id'] ?? 0;
    }

    $conn = getDBConnection();
    $userId = (int) $userId;

    $sql = "SELECT * FROM user_nav_permissions WHERE user_id = $userId";
    $result = $conn->query($sql);

    $permissions = [
        'dashboard' => 1,
        'search' => 1,
        'history' => 1,
        'profile' => 1,
        'support' => 1
    ];

    if ($result && $result->num_rows > 0) {
        $permissions = $result->fetch_assoc();
    }

    $conn->close();
    return $permissions;
}

/**
 * Check if user has permission for a nav item
 */
function hasNavPermission($navItem)
{
    if (isAdmin()) {
        return true; // Admins have all permissions
    }

    $permissions = getUserNavPermissions();
    return isset($permissions[$navItem]) && $permissions[$navItem] == 1;
}

/**
 * Sanitize input
 */
function sanitizeInput($data)
{
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data);
    return $data;
}
