<?php
/**
 * Bulk Contact Processor - Simplified
 * Just marks contacts as processed in search_logs
 */

ob_start();
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/api.php';
ob_end_clean();

set_time_limit(0);
ini_set('display_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json');

try {
    if (isset($_GET['status'])) {
        echo json_encode(getProcessingStatus());
        exit;
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $batchSize = isset($_POST['batch_size']) ? (int) $_POST['batch_size'] : 10;
        $userId = 1; // Default user ID for bulk processing
        processContacts($batchSize, $userId);
        exit;
    }

    echo json_encode(['success' => false, 'message' => 'Invalid request']);

} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

function processContacts($batchSize, $userId)
{
    $conn = getDBConnection();

    // Get unprocessed contacts
    $sql = "SELECT * FROM imported_contacts 
            WHERE phone NOT IN (
                SELECT mobile_number FROM search_logs WHERE user_id = $userId
            )
            LIMIT $batchSize";

    $result = $conn->query($sql);

    if (!$result || $result->num_rows === 0) {
        echo json_encode([
            'success' => true,
            'message' => 'No contacts to process',
            'processed' => 0,
            'success_count' => 0,
            'failed_count' => 0,
            'remaining' => 0
        ]);
        $conn->close();
        return;
    }

    $processed = 0;
    $success = 0;
    $failed = 0;

    while ($contact = $result->fetch_assoc()) {
        $phone = $contact['phone'];

        // Fetch from API using existing function
        $apiData = fetchFromAPI($phone);

        if ($apiData) {
            // Store in search_logs (same as manual search)
            $dataJson = is_array($apiData) ? json_encode($apiData) : $apiData;
            $dataJson = $conn->real_escape_string($dataJson);
            $phone = $conn->real_escape_string($phone);

            $insertSql = "INSERT INTO search_logs (user_id, mobile_number, data, source, search_date) 
                          VALUES ($userId, '$phone', '$dataJson', 'API', NOW())";

            if ($conn->query($insertSql)) {
                $success++;
            } else {
                $failed++;
            }
        } else {
            $failed++;
        }

        $processed++;
        usleep(500000); // 0.5 second delay
    }

    // Get remaining count
    $remainingSql = "SELECT COUNT(*) as count FROM imported_contacts 
                     WHERE phone NOT IN (
                         SELECT mobile_number FROM search_logs WHERE user_id = $userId
                     )";
    $remainingResult = $conn->query($remainingSql);
    $remaining = 0;
    if ($remainingResult) {
        $remaining = $remainingResult->fetch_assoc()['count'];
    }

    $conn->close();

    echo json_encode([
        'success' => true,
        'message' => "Processed $processed contacts",
        'processed' => $processed,
        'success_count' => $success,
        'failed_count' => $failed,
        'remaining' => $remaining
    ]);
}

function getProcessingStatus()
{
    try {
        $conn = getDBConnection();
        $userId = 1; // Default user ID

        // Total imported
        $totalSql = "SELECT COUNT(*) as count FROM imported_contacts";
        $totalResult = $conn->query($totalSql);
        $total = $totalResult ? $totalResult->fetch_assoc()['count'] : 0;

        // Already processed (in search_logs)
        $processedSql = "SELECT COUNT(*) as count FROM imported_contacts 
                         WHERE phone IN (
                             SELECT mobile_number FROM search_logs WHERE user_id = $userId
                         )";
        $processedResult = $conn->query($processedSql);
        $processed = $processedResult ? $processedResult->fetch_assoc()['count'] : 0;

        // Remaining
        $remaining = $total - $processed;

        $conn->close();

        return [
            'total' => $total,
            'processed' => $processed,
            'remaining' => $remaining,
            'percentage' => $total > 0 ? round(($processed / $total) * 100, 2) : 0
        ];

    } catch (Exception $e) {
        return [
            'total' => 0,
            'processed' => 0,
            'remaining' => 0,
            'percentage' => 0,
            'error' => $e->getMessage()
        ];
    }
}
