<?php
/**
 * VCF Contact Processor
 * Parses VCF file and stores contacts in database
 */

require_once __DIR__ . '/config/database.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

if (!isset($_FILES['vcf_file'])) {
    echo json_encode(['success' => false, 'message' => 'No file uploaded']);
    exit;
}

$file = $_FILES['vcf_file'];

// Validate file
if ($file['error'] !== UPLOAD_ERR_OK) {
    echo json_encode(['success' => false, 'message' => 'File upload error']);
    exit;
}

if (!str_ends_with(strtolower($file['name']), '.vcf')) {
    echo json_encode(['success' => false, 'message' => 'Only .vcf files are allowed']);
    exit;
}

// Read file content
$content = file_get_contents($file['tmp_name']);

if (!$content) {
    echo json_encode(['success' => false, 'message' => 'Failed to read file']);
    exit;
}

// Parse VCF
$contacts = parseVCF($content);

// Create table if not exists
$conn = getDBConnection();
createContactsTable($conn);

// Import contacts
$stats = importContacts($conn, $contacts);

$conn->close();

echo json_encode([
    'success' => true,
    'message' => "Successfully imported {$stats['imported']} contacts!",
    'stats' => $stats
]);

/**
 * Parse VCF file content
 */
function parseVCF($content)
{
    $contacts = [];
    $lines = explode("\n", $content);

    $currentContact = null;

    foreach ($lines as $line) {
        $line = trim($line);

        if (empty($line))
            continue;

        // Start of new contact
        if (stripos($line, 'BEGIN:VCARD') === 0) {
            $currentContact = [
                'name' => '',
                'phones' => [],
                'emails' => [],
                'organization' => ''
            ];
        }

        // End of contact
        elseif (stripos($line, 'END:VCARD') === 0) {
            if ($currentContact && !empty($currentContact['phones'])) {
                $contacts[] = $currentContact;
            }
            $currentContact = null;
        }

        // Parse fields
        elseif ($currentContact) {
            // Name
            if (stripos($line, 'FN:') === 0) {
                $currentContact['name'] = substr($line, 3);
            } elseif (stripos($line, 'N:') === 0 && empty($currentContact['name'])) {
                $parts = explode(';', substr($line, 2));
                $currentContact['name'] = trim($parts[1] . ' ' . $parts[0]);
            }

            // Phone
            elseif (stripos($line, 'TEL') === 0) {
                $phone = extractValue($line);
                if ($phone) {
                    $currentContact['phones'][] = cleanPhone($phone);
                }
            }

            // Email
            elseif (stripos($line, 'EMAIL') === 0) {
                $email = extractValue($line);
                if ($email) {
                    $currentContact['emails'][] = $email;
                }
            }

            // Organization
            elseif (stripos($line, 'ORG:') === 0) {
                $currentContact['organization'] = substr($line, 4);
            }
        }
    }

    return $contacts;
}

/**
 * Extract value from VCF line
 */
function extractValue($line)
{
    $parts = explode(':', $line, 2);
    return isset($parts[1]) ? trim($parts[1]) : '';
}

/**
 * Clean phone number
 */
function cleanPhone($phone)
{
    // Remove all non-numeric characters except +
    $phone = preg_replace('/[^0-9+]/', '', $phone);

    // Remove country code if present
    $phone = preg_replace('/^\+91/', '', $phone);
    $phone = preg_replace('/^91/', '', $phone);

    // Get last 10 digits
    if (strlen($phone) > 10) {
        $phone = substr($phone, -10);
    }

    return $phone;
}

/**
 * Create contacts table
 */
function createContactsTable($conn)
{
    $sql = "CREATE TABLE IF NOT EXISTS imported_contacts (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255),
        phone VARCHAR(20),
        email VARCHAR(255),
        organization VARCHAR(255),
        imported_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        INDEX(phone)
    )";

    $conn->query($sql);
}

/**
 * Import contacts to database
 */
function importContacts($conn, $contacts)
{
    $total = count($contacts);
    $imported = 0;
    $skipped = 0;
    $withPhone = 0;

    foreach ($contacts as $contact) {
        if (empty($contact['phones'])) {
            $skipped++;
            continue;
        }

        $withPhone++;

        foreach ($contact['phones'] as $phone) {
            // Skip if invalid
            if (strlen($phone) != 10 || !ctype_digit($phone)) {
                continue;
            }

            // Check if already exists
            $checkSql = "SELECT id FROM imported_contacts WHERE phone = '$phone'";
            $result = $conn->query($checkSql);

            if ($result && $result->num_rows > 0) {
                $skipped++;
                continue;
            }

            // Insert
            $name = $conn->real_escape_string($contact['name']);
            $email = !empty($contact['emails']) ? $conn->real_escape_string($contact['emails'][0]) : '';
            $org = $conn->real_escape_string($contact['organization']);

            $insertSql = "INSERT INTO imported_contacts (name, phone, email, organization) 
                         VALUES ('$name', '$phone', '$email', '$org')";

            if ($conn->query($insertSql)) {
                $imported++;
            }
        }
    }

    return [
        'total' => $total,
        'with_phone' => $withPhone,
        'imported' => $imported,
        'skipped' => $skipped
    ];
}
