<?php
session_start();
require_once 'db.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

$user_id = $_SESSION['user_id'];

// Get mobile number from POST
$mobile = isset($_POST['mobile']) ? sanitize_input($_POST['mobile']) : '';

// Validate mobile number
if (empty($mobile) || !preg_match('/^[0-9]{10}$/', $mobile)) {
    echo json_encode([
        'success' => false,
        'error_title' => 'Invalid Number',
        'message' => 'Please enter a valid 10-digit mobile number'
    ]);
    exit();
}

// STEP 1: Check database first
$db_result = get_mobile_info_from_db($mobile);

if ($db_result) {
    // Data found in database (can be single or multiple records)
    log_search_history($user_id, $mobile, 'found_db');

    // Format all records
    $formatted_records = [];

    foreach ($db_result as $record) {
        // Parse extra_data JSON if available
        $extra_data = [];
        if (isset($record['extra_data']) && !empty($record['extra_data'])) {
            $extra_data = json_decode($record['extra_data'], true) ?: [];
        }

        $formatted_records[] = [
            'mobile' => $record['mobile'],
            'name' => $record['name'],
            'fname' => $extra_data['fname'] ?? null,
            'address' => $extra_data['address'] ?? null,
            'alt_mobile' => $extra_data['alt_mobile'] ?? null,
            'aadhaar_id' => $extra_data['aadhaar_id'] ?? null,
            'circle' => $record['circle'],
            'operator' => $record['operator']
        ];
    }

    echo json_encode([
        'success' => true,
        'source' => 'database',
        'count' => count($formatted_records),
        'data' => $formatted_records
    ]);
    exit();
}

// STEP 2: Data not in database, fetch from API
$api_result = fetch_from_api($mobile);

if ($api_result['success']) {
    // Save to database
    save_mobile_info_to_db(
        $mobile,
        $api_result['data']['name'] ?? null,
        $api_result['data']['circle'] ?? null,
        $api_result['data']['operator'] ?? null,
        json_encode($api_result['data'])
    );

    log_search_history($user_id, $mobile, 'found_api');

    echo json_encode([
        'success' => true,
        'source' => 'api',
        'data' => [
            'mobile' => $mobile,
            'name' => $api_result['data']['name'] ?? 'N/A',
            'circle' => $api_result['data']['circle'] ?? 'N/A',
            'operator' => $api_result['data']['operator'] ?? 'N/A'
        ]
    ]);
    exit();
} else {
    // API is down or data not found
    if ($api_result['api_down']) {
        log_search_history($user_id, $mobile, 'api_down');
        echo json_encode([
            'success' => false,
            'error_title' => 'Service Unavailable',
            'message' => 'Data not available right now. Please try again later.'
        ]);
    } else {
        log_search_history($user_id, $mobile, 'not_found');
        echo json_encode([
            'success' => false,
            'error_title' => 'Not Found',
            'message' => 'No information found for this mobile number.'
        ]);
    }
    exit();
}

/**
 * Get mobile info from database
 * Checks both mobile_info and searches tables
 * Returns ALL matching records (array of records)
 */
function get_mobile_info_from_db($mobile)
{
    global $conn;
    $mobile = mysqli_real_escape_string($conn, $mobile);
    $all_records = [];

    // First check mobile_info table
    $sql = "SELECT * FROM mobile_info WHERE mobile = '$mobile'";
    $result = mysqli_query($conn, $sql);

    if ($result && mysqli_num_rows($result) > 0) {
        while ($row = mysqli_fetch_assoc($result)) {
            $all_records[] = $row;
        }
    }

    // Also check searches table for this number
    $search_sql = "SELECT * FROM searches WHERE search_type = 'mobile' AND query_value = '$mobile' LIMIT 1";
    $search_result = mysqli_query($conn, $search_sql);

    if ($search_result && mysqli_num_rows($search_result) > 0) {
        $search_data = mysqli_fetch_assoc($search_result);
        $json_response = json_decode($search_data['json_response'], true);

        // Extract ALL mobile records from JSON array
        if (isset($json_response['data']['data']) && is_array($json_response['data']['data'])) {
            foreach ($json_response['data']['data'] as $record) {
                // Check if this record's mobile matches our search
                $record_mobile = $record['mobile'] ?? '';

                // Add record if mobile matches OR if searching for alt number
                if ($record_mobile === $mobile || ($record['alt'] ?? '') === $mobile) {
                    $all_records[] = [
                        'mobile' => $record['mobile'] ?? $mobile,
                        'name' => $record['name'] ?? 'N/A',
                        'circle' => $record['circle'] ?? 'N/A',
                        'operator' => extract_operator($record['circle'] ?? ''),
                        'extra_data' => json_encode([
                            'fname' => $record['fname'] ?? null,
                            'address' => $record['address'] ?? null,
                            'alt_mobile' => $record['alt'] ?? null,
                            'aadhaar_id' => $record['id'] ?? null,
                            'source' => 'searches_table'
                        ])
                    ];
                }
            }
        }
    }

    return count($all_records) > 0 ? $all_records : null;
}

/**
 * Extract operator name from circle
 */
function extract_operator($circle)
{
    if (empty($circle))
        return 'N/A';

    $parts = explode(' ', $circle);
    return $parts[0];
}

/**
 * Fetch data from external API
 */
function fetch_from_api($mobile)
{
    // API Configuration
    $api_url = "https://api.example.com/mobile-info?number=" . $mobile;
    $fallback_api_url = "https://old-api.example.com/lookup?mobile=" . $mobile;

    // Try primary API first
    $result = make_api_request($api_url);

    if ($result['success']) {
        return $result;
    }

    // Try fallback API
    $result = make_api_request($fallback_api_url);

    return $result;
}

/**
 * Make API request with timeout handling
 */
function make_api_request($url)
{
    $ch = curl_init();

    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5); // 5 second timeout
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'User-Agent: MobileSearchApp/1.0',
        'Accept: application/json'
    ]);

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);

    curl_close($ch);

    // Check for cURL errors
    if ($curl_error) {
        return [
            'success' => false,
            'api_down' => true,
            'message' => 'API connection failed'
        ];
    }

    // Check HTTP status
    if ($http_code !== 200) {
        return [
            'success' => false,
            'api_down' => true,
            'message' => 'API returned error: ' . $http_code
        ];
    }

    // Parse JSON response
    $data = json_decode($response, true);

    if (!$data) {
        return [
            'success' => false,
            'api_down' => false,
            'message' => 'Invalid API response'
        ];
    }

    // Check if data exists in response
    if (isset($data['error']) || empty($data)) {
        return [
            'success' => false,
            'api_down' => false,
            'message' => 'No data found'
        ];
    }

    return [
        'success' => true,
        'data' => $data
    ];
}

/**
 * Save mobile info to database
 */
function save_mobile_info_to_db($mobile, $name, $circle, $operator, $extra_data)
{
    global $conn;

    $mobile = mysqli_real_escape_string($conn, $mobile);
    $name = $name ? "'" . mysqli_real_escape_string($conn, $name) . "'" : 'NULL';
    $circle = $circle ? "'" . mysqli_real_escape_string($conn, $circle) . "'" : 'NULL';
    $operator = $operator ? "'" . mysqli_real_escape_string($conn, $operator) . "'" : 'NULL';
    $extra_data = $extra_data ? "'" . mysqli_real_escape_string($conn, $extra_data) . "'" : 'NULL';

    $sql = "INSERT INTO mobile_info (mobile, name, circle, operator, extra_data) 
            VALUES ('$mobile', $name, $circle, $operator, $extra_data)
            ON DUPLICATE KEY UPDATE 
            name = $name, 
            circle = $circle, 
            operator = $operator, 
            extra_data = $extra_data";

    return mysqli_query($conn, $sql);
}

/**
 * Log search history
 */
function log_search_history($user_id, $mobile, $result_type)
{
    global $conn;

    $user_id = (int) $user_id;
    $mobile = mysqli_real_escape_string($conn, $mobile);
    $result_type = mysqli_real_escape_string($conn, $result_type);

    $sql = "INSERT INTO search_history (user_id, mobile, search_result) 
            VALUES ($user_id, '$mobile', '$result_type')";

    return mysqli_query($conn, $sql);
}
