<?php
// Chat Widget - Include this in pages where you want chat
if (!isset($_SESSION['user_id'])) {
    return;
}
?>

<style>
    /* Floating Chat Button - Hidden (using navbar instead) */
    .chat-float-btn {
        display: none !important;
    }

    /* Chat Window */
    .chat-window {
        position: fixed;
        bottom: 90px;
        right: 20px;
        width: 90%;
        max-width: 380px;
        height: 500px;
        background: white;
        border-radius: 20px;
        box-shadow: 0 8px 40px rgba(0, 0, 0, 0.2);
        display: none;
        flex-direction: column;
        z-index: 1000;
        overflow: hidden;
    }

    .chat-window.active {
        display: flex;
    }

    /* Chat Header */
    .chat-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 16px 20px;
        display: flex;
        align-items: center;
        justify-content: space-between;
    }

    .chat-header-left {
        display: flex;
        align-items: center;
        gap: 12px;
    }

    .chat-avatar {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        background: rgba(255, 255, 255, 0.2);
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 18px;
    }

    .chat-header-info h3 {
        font-size: 16px;
        font-weight: 600;
        margin: 0;
    }

    .chat-header-info p {
        font-size: 12px;
        opacity: 0.9;
        margin: 2px 0 0 0;
    }

    .chat-close-btn {
        background: none;
        border: none;
        color: white;
        cursor: pointer;
        padding: 4px;
    }

    /* Chat Mode Toggle */
    .chat-mode-toggle {
        display: flex;
        gap: 8px;
        padding: 12px 16px;
        background: #F5F5F7;
        border-bottom: 1px solid #E5E5EA;
    }

    .mode-btn {
        flex: 1;
        padding: 8px 12px;
        border: none;
        border-radius: 8px;
        background: white;
        color: #8E8E93;
        font-size: 13px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s;
    }

    .mode-btn.active {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
    }

    /* Chat Messages */
    .chat-messages {
        flex: 1;
        overflow-y: auto;
        padding: 16px;
        background: #F5F5F7;
    }

    .chat-message {
        margin-bottom: 12px;
        display: flex;
        gap: 8px;
    }

    .chat-message.user {
        flex-direction: row-reverse;
    }

    .message-avatar {
        width: 32px;
        height: 32px;
        border-radius: 50%;
        background: #667eea;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 14px;
        flex-shrink: 0;
    }

    .chat-message.user .message-avatar {
        background: #34C759;
    }

    .message-bubble {
        max-width: 70%;
        padding: 10px 14px;
        border-radius: 16px;
        background: white;
        color: #000;
        font-size: 14px;
        line-height: 1.4;
    }

    .chat-message.user .message-bubble {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
    }

    .message-time {
        font-size: 11px;
        color: #8E8E93;
        margin-top: 4px;
    }

    /* Chat Input */
    .chat-input-area {
        padding: 12px 16px;
        background: white;
        border-top: 1px solid #E5E5EA;
        display: flex;
        gap: 8px;
        align-items: center;
    }

    .chat-input {
        flex: 1;
        padding: 10px 14px;
        border: 1px solid #E5E5EA;
        border-radius: 20px;
        font-size: 14px;
        outline: none;
    }

    .chat-input:focus {
        border-color: #667eea;
    }

    .chat-send-btn {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border: none;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        transition: all 0.2s;
    }

    .chat-send-btn:hover {
        transform: scale(1.05);
    }

    .chat-send-btn svg {
        width: 20px;
        height: 20px;
        fill: white;
    }

    /* Typing Indicator */
    .typing-indicator {
        display: none;
        padding: 10px 14px;
        background: white;
        border-radius: 16px;
        width: fit-content;
    }

    .typing-indicator.active {
        display: block;
    }

    .typing-dot {
        display: inline-block;
        width: 8px;
        height: 8px;
        border-radius: 50%;
        background: #8E8E93;
        margin: 0 2px;
        animation: typing 1.4s infinite;
    }

    .typing-dot:nth-child(2) {
        animation-delay: 0.2s;
    }

    .typing-dot:nth-child(3) {
        animation-delay: 0.4s;
    }

    @keyframes typing {

        0%,
        60%,
        100% {
            transform: translateY(0);
        }

        30% {
            transform: translateY(-10px);
        }
    }
</style>

<!-- Floating Chat Button -->
<div class="chat-float-btn" onclick="toggleChat()">
    <svg viewBox="0 0 24 24">
        <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z" />
    </svg>
    <div class="chat-badge" id="chatBadge" style="display: none;">1</div>
</div>

<!-- Chat Window -->
<div class="chat-window" id="chatWindow">
    <!-- Header -->
    <div class="chat-header">
        <div class="chat-header-left">
            <div class="chat-avatar">🤖</div>
            <div class="chat-header-info">
                <h3 id="chatTitle">AI Assistant</h3>
                <p id="chatStatus">Online</p>
            </div>
        </div>
        <button class="chat-close-btn" onclick="toggleChat()">
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M18 6L6 18M6 6l12 12" />
            </svg>
        </button>
    </div>

    <!-- Mode Toggle -->
    <div class="chat-mode-toggle">
        <button class="mode-btn active" data-mode="ai" onclick="switchMode('ai')">
            🤖 AI Chat
        </button>
        <button class="mode-btn" data-mode="admin" onclick="switchMode('admin')">
            👨‍💼 Admin
        </button>
    </div>

    <!-- Messages -->
    <div class="chat-messages" id="chatMessages">
        <div class="chat-message">
            <div class="message-avatar">🤖</div>
            <div>
                <div class="message-bubble">
                    Hi! I'm your AI assistant. How can I help you today?
                </div>
            </div>
        </div>
    </div>

    <!-- Input Area -->
    <div class="chat-input-area">
        <input type="text" class="chat-input" id="chatInput" placeholder="Type a message..."
            onkeypress="handleEnter(event)">
        <button class="chat-send-btn" onclick="sendMessage()">
            <svg viewBox="0 0 24 24">
                <path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z" />
            </svg>
        </button>
    </div>
</div>

<script>
    let currentMode = 'ai';

    function toggleChat() {
        const chatWindow = document.getElementById('chatWindow');
        chatWindow.classList.toggle('active');

        if (chatWindow.classList.contains('active')) {
            loadChatHistory();
            document.getElementById('chatInput').focus();
        }
    }

    function switchMode(mode) {
        currentMode = mode;

        // Update buttons
        document.querySelectorAll('.mode-btn').forEach(btn => {
            btn.classList.remove('active');
            if (btn.dataset.mode === mode) {
                btn.classList.add('active');
            }
        });

        // Update header
        const title = mode === 'ai' ? 'AI Assistant' : 'Chat with Admin';
        const avatar = mode === 'ai' ? '🤖' : '👨‍💼';

        document.getElementById('chatTitle').textContent = title;
        document.querySelector('.chat-avatar').textContent = avatar;
    }

    function handleEnter(event) {
        if (event.key === 'Enter') {
            sendMessage();
        }
    }

    async function sendMessage() {
        const input = document.getElementById('chatInput');
        const message = input.value.trim();

        if (!message) return;

        // Add user message to chat
        addMessage(message, true);
        input.value = '';

        // Show typing indicator
        showTyping();

        try {
            const formData = new FormData();
            formData.append('action', 'send');
            formData.append('message', message);
            formData.append('mode', currentMode);

            const response = await fetch('api/chat_api.php', {
                method: 'POST',
                body: formData
            });

            if (!response.ok) {
                throw new Error('Network response was not ok');
            }

            const data = await response.json();

            hideTyping();

            if (data.success) {
                if (data.response) {
                    addMessage(data.response, false);
                } else if (data.message) {
                    addMessage(data.message, false, true);
                }
            } else {
                addMessage(data.error || 'Sorry, something went wrong. Please try again.', false);
            }
        } catch (error) {
            console.error('Chat error:', error);
            hideTyping();
            addMessage('Connection error. Please check if chat API is accessible.', false);
        }
    }

    function addMessage(text, isUser = false, isAdmin = false) {
        const messagesDiv = document.getElementById('chatMessages');
        const messageDiv = document.createElement('div');
        messageDiv.className = 'chat-message' + (isUser ? ' user' : '');

        const avatar = isUser ? '👤' : (currentMode === 'admin' || isAdmin ? '👨‍💼' : '🤖');

        messageDiv.innerHTML = `
        <div class="message-avatar">${avatar}</div>
        <div>
            <div class="message-bubble">${text}</div>
            <div class="message-time">${new Date().toLocaleTimeString('en-US', { hour: '2-digit', minute: '2-digit' })}</div>
        </div>
    `;

        messagesDiv.appendChild(messageDiv);
        messagesDiv.scrollTop = messagesDiv.scrollHeight;
    }

    function showTyping() {
        const messagesDiv = document.getElementById('chatMessages');
        const typingDiv = document.createElement('div');
        typingDiv.id = 'typingIndicator';
        typingDiv.className = 'chat-message';
        typingDiv.innerHTML = `
        <div class="message-avatar">🤖</div>
        <div class="typing-indicator active">
            <span class="typing-dot"></span>
            <span class="typing-dot"></span>
            <span class="typing-dot"></span>
        </div>
    `;
        messagesDiv.appendChild(typingDiv);
        messagesDiv.scrollTop = messagesDiv.scrollHeight;
    }

    function hideTyping() {
        const typing = document.getElementById('typingIndicator');
        if (typing) typing.remove();
    }

    async function loadChatHistory() {
        try {
            const response = await fetch('api/chat_api.php?action=history');
            const data = await response.json();

            if (data.success && data.messages.length > 0) {
                const messagesDiv = document.getElementById('chatMessages');
                messagesDiv.innerHTML = '';

                data.messages.forEach(msg => {
                    addMessage(msg.message, msg.is_user == 1, msg.is_admin == 1);
                });
            }
        } catch (error) {
            console.error('Failed to load chat history:', error);
        }
    }
</script>