<?php
session_start();
require_once 'db.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit();
}

$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Messages</title>
    <link rel="stylesheet" href="style.css">
    <style>
        .messages-container {
            display: flex;
            height: 100vh;
            background: #F5F5F7;
        }

        /* Conversations List */
        .conversations-sidebar {
            width: 100%;
            max-width: 380px;
            background: white;
            border-right: 1px solid #E5E5EA;
            display: flex;
            flex-direction: column;
        }

        .messages-header {
            padding: 20px;
            border-bottom: 1px solid #E5E5EA;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .messages-header h2 {
            font-size: 24px;
            font-weight: 700;
            margin: 0;
        }

        .new-message-btn {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
        }

        .conversations-list {
            flex: 1;
            overflow-y: auto;
        }

        .conversation-item {
            padding: 16px 20px;
            border-bottom: 1px solid #F5F5F7;
            cursor: pointer;
            transition: background 0.2s;
            display: flex;
            gap: 12px;
            align-items: center;
        }

        .conversation-item:hover {
            background: #F5F5F7;
        }

        .conversation-item.active {
            background: #F0F0F5;
            border-left: 3px solid #667eea;
        }

        .conv-avatar {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: #667eea;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 18px;
            font-weight: 600;
            flex-shrink: 0;
        }

        .conv-avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }

        .conv-info {
            flex: 1;
            min-width: 0;
        }

        .conv-name {
            font-weight: 600;
            font-size: 15px;
            margin-bottom: 4px;
        }

        .conv-last-message {
            font-size: 13px;
            color: #8E8E93;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }

        .conv-badge {
            background: #FF3B30;
            color: white;
            border-radius: 12px;
            padding: 2px 8px;
            font-size: 11px;
            font-weight: 600;
        }

        /* Chat Area */
        .chat-area {
            flex: 1;
            display: flex;
            flex-direction: column;
            background: white;
        }

        .chat-header {
            padding: 20px;
            border-bottom: 1px solid #E5E5EA;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .chat-messages {
            flex: 1;
            overflow-y: auto;
            padding: 20px;
            background: #F5F5F7;
        }

        .message {
            margin-bottom: 16px;
            display: flex;
            gap: 8px;
        }

        .message.sent {
            flex-direction: row-reverse;
        }

        .message-bubble {
            max-width: 70%;
            padding: 12px 16px;
            border-radius: 16px;
            background: white;
            font-size: 14px;
            line-height: 1.4;
        }

        .message.sent .message-bubble {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .message-time {
            font-size: 11px;
            color: #8E8E93;
            margin-top: 4px;
        }

        .chat-input-area {
            padding: 16px 20px;
            background: white;
            border-top: 1px solid #E5E5EA;
            display: flex;
            gap: 12px;
        }

        .chat-input {
            flex: 1;
            padding: 12px 16px;
            border: 1px solid #E5E5EA;
            border-radius: 24px;
            font-size: 14px;
            outline: none;
        }

        .send-btn {
            width: 44px;
            height: 44px;
            border-radius: 50%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            color: white;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .empty-state {
            flex: 1;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            color: #8E8E93;
        }

        @media (max-width: 768px) {
            .conversations-sidebar {
                max-width: 100%;
            }

            .chat-area {
                display: none;
            }

            .chat-area.active {
                display: flex;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                z-index: 1000;
            }
        }
    </style>
</head>

<body>
    <div class="messages-container">
        <!-- Conversations Sidebar -->
        <div class="conversations-sidebar" id="conversationsSidebar">
            <div class="messages-header">
                <h2>💬 Messages</h2>
                <button class="new-message-btn" onclick="showNewMessageModal()">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M12 5v14M5 12h14" />
                    </svg>
                </button>
            </div>
            <div class="conversations-list" id="conversationsList">
                <!-- Conversations will load here -->
            </div>
        </div>

        <!-- Chat Area -->
        <div class="chat-area" id="chatArea">
            <div class="empty-state">
                <svg width="80" height="80" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z" />
                </svg>
                <h3 style="margin: 16px 0 8px 0;">Select a conversation</h3>
                <p>Choose a conversation to start messaging</p>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation -->
    <?php include 'navbar.php'; ?>

    <script src="script.js"></script>
    <script>
        let currentChatUserId = null;
        let messagePolling = null;

        // Load conversations
        async function loadConversations() {
            try {
                const response = await fetch('api/messaging_api.php?action=get_conversations');
                const data = await response.json();

                if (data.success) {
                    displayConversations(data.conversations);
                }
            } catch (error) {
                console.error('Error loading conversations:', error);
            }
        }

        function displayConversations(conversations) {
            const list = document.getElementById('conversationsList');

            if (conversations.length === 0) {
                list.innerHTML = '<div style="padding: 40px 20px; text-align: center; color: #8E8E93;">No conversations yet</div>';
                return;
            }

            list.innerHTML = conversations.map(conv => `
                <div class="conversation-item ${currentChatUserId == conv.other_user_id ? 'active' : ''}" 
                     onclick="openChat(${conv.other_user_id}, '${conv.other_user_name}')">
                    <div class="conv-avatar">
                        ${conv.profile_picture ? `<img src="${conv.profile_picture}" alt="">` : conv.other_user_name.charAt(0).toUpperCase()}
                    </div>
                    <div class="conv-info">
                        <div class="conv-name">${conv.other_user_name}</div>
                        <div class="conv-last-message">${conv.last_message || 'No messages yet'}</div>
                    </div>
                    ${conv.unread_count > 0 ? `<span class="conv-badge">${conv.unread_count}</span>` : ''}
                </div>
            `).join('');
        }

        async function openChat(userId, userName) {
            currentChatUserId = userId;

            const chatArea = document.getElementById('chatArea');
            chatArea.innerHTML = `
                <div class="chat-header">
                    <button onclick="closeChat()" style="background: none; border: none; cursor: pointer; display: none;" id="backBtn">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M19 12H5M12 19l-7-7 7-7"/>
                        </svg>
                    </button>
                    <div class="conv-avatar">${userName.charAt(0).toUpperCase()}</div>
                    <h3>${userName}</h3>
                </div>
                <div class="chat-messages" id="chatMessages"></div>
                <div class="chat-input-area">
                    <input type="text" class="chat-input" id="messageInput" placeholder="Type a message..." onkeypress="handleEnter(event)">
                    <button class="send-btn" onclick="sendMessage()">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z"/>
                        </svg>
                    </button>
                </div>
            `;

            chatArea.classList.add('active');
            if (window.innerWidth <= 768) {
                document.getElementById('backBtn').style.display = 'block';
            }

            loadMessages(userId);
            startMessagePolling();
        }

        async function loadMessages(userId) {
            try {
                const response = await fetch(`api/messaging_api.php?action=get_messages&user_id=${userId}`);
                const data = await response.json();

                if (data.success) {
                    displayMessages(data.messages);
                }
            } catch (error) {
                console.error('Error loading messages:', error);
            }
        }

        function displayMessages(messages) {
            const container = document.getElementById('chatMessages');

            container.innerHTML = messages.map(msg => `
                <div class="message ${msg.sender_id == <?php echo $user_id; ?> ? 'sent' : 'received'}">
                    <div class="message-bubble">
                        ${msg.message}
                        <div class="message-time">${new Date(msg.created_at).toLocaleTimeString('en-US', { hour: '2-digit', minute: '2-digit' })}</div>
                    </div>
                </div>
            `).join('');

            container.scrollTop = container.scrollHeight;
        }

        async function sendMessage() {
            const input = document.getElementById('messageInput');
            const message = input.value.trim();

            if (!message || !currentChatUserId) return;

            try {
                const formData = new FormData();
                formData.append('action', 'send_message');
                formData.append('receiver_id', currentChatUserId);
                formData.append('message', message);

                const response = await fetch('api/messaging_api.php', {
                    method: 'POST',
                    body: formData
                });

                const data = await response.json();

                if (data.success) {
                    input.value = '';
                    loadMessages(currentChatUserId);
                    loadConversations();
                }
            } catch (error) {
                console.error('Error sending message:', error);
            }
        }

        function handleEnter(event) {
            if (event.key === 'Enter') {
                sendMessage();
            }
        }

        function startMessagePolling() {
            if (messagePolling) clearInterval(messagePolling);
            messagePolling = setInterval(() => {
                if (currentChatUserId) {
                    loadMessages(currentChatUserId);
                }
                loadConversations();
            }, 3000);
        }

        function closeChat() {
            document.getElementById('chatArea').classList.remove('active');
        }

        function showNewMessageModal() {
            showToast('Feature coming soon!', 'info');
        }

        // Initialize
        loadConversations();
        setInterval(loadConversations, 10000);
    </script>
</body>

</html>