/**
 * Mobile App JavaScript
 * Form validation, animations, and interactions
 */

// ==========================================
// TOAST NOTIFICATION SYSTEM (GLOBAL)
// ==========================================

/**
 * Show toast notification
 */
function showToast(message, type = 'info', duration = 3000) {
    // Create toast container if it doesn't exist
    let container = document.querySelector('.toast-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'toast-container';
        document.body.appendChild(container);
    }

    // Create toast element
    const toast = document.createElement('div');
    toast.className = `toast ${type}`;

    // Icon based on type
    let icon = '';
    let title = '';

    if (type === 'success') {
        title = 'Success';
        icon = '<svg viewBox="0 0 24 24" fill="#34C759"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/></svg>';
    } else if (type === 'error') {
        title = 'Error';
        icon = '<svg viewBox="0 0 24 24" fill="#FF3B30"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"/></svg>';
    } else {
        title = 'Info';
        icon = '<svg viewBox="0 0 24 24" fill="#007AFF"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z"/></svg>';
    }

    toast.innerHTML = `
        <div class="toast-icon">${icon}</div>
        <div class="toast-content">
            <div class="toast-title">${title}</div>
            <div class="toast-message">${message}</div>
        </div>
        <button class="toast-close" onclick="this.parentElement.remove()">
            <svg viewBox="0 0 24 24" fill="currentColor">
                <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/>
            </svg>
        </button>
    `;

    container.appendChild(toast);

    // Auto remove after duration
    setTimeout(() => {
        toast.style.animation = 'fadeOut 0.3s ease forwards';
        setTimeout(() => {
            toast.remove();
            if (container.children.length === 0) {
                container.remove();
            }
        }, 300);
    }, duration);
}

// ==========================================
// DARK MODE TOGGLE
// ==========================================

// Check saved theme preference
const savedTheme = localStorage.getItem('theme') || 'light';
document.documentElement.setAttribute('data-theme', savedTheme);

// Toggle theme function
function toggleTheme() {
    const currentTheme = document.documentElement.getAttribute('data-theme');
    const newTheme = currentTheme === 'light' ? 'dark' : 'light';

    document.documentElement.setAttribute('data-theme', newTheme);
    localStorage.setItem('theme', newTheme);

    // Haptic feedback
    if (navigator.vibrate) {
        navigator.vibrate(10);
    }
}

// ==========================================
// DOM CONTENT LOADED
// ==========================================

document.addEventListener('DOMContentLoaded', function () {

    // Theme toggle button removed - dark mode still works via localStorage
    // Users can toggle via browser settings or we can add it back later

    // Convert alerts to toasts
    const successAlerts = document.querySelectorAll('.alert-success');
    const errorAlerts = document.querySelectorAll('.alert-error');

    successAlerts.forEach(alert => {
        const message = alert.textContent.trim();
        showToast(message, 'success');
        alert.style.display = 'none';
    });

    errorAlerts.forEach(alert => {
        const message = alert.textContent.trim();
        showToast(message, 'error');
        alert.style.display = 'none';
    });

    // Form submission with loading state
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        form.addEventListener('submit', function (e) {
            const submitBtn = form.querySelector('button[type="submit"]');
            if (submitBtn && !submitBtn.classList.contains('loading')) {
                submitBtn.classList.add('loading');
                setTimeout(() => {
                    submitBtn.classList.remove('loading');
                }, 5000);
            }
        });
    });

    // Haptic feedback for buttons
    const buttons = document.querySelectorAll('button, .btn, a.nav-item');
    buttons.forEach(button => {
        button.addEventListener('click', function () {
            if (navigator.vibrate) {
                navigator.vibrate(10);
            }
        });
    });
});
