<?php
require_once __DIR__ . '/../models/Alert.php';
require_once __DIR__ . '/../libraries/SimpleMailer.php';

class Notification {
    private $alertModel;
    private $mailer;
    
    public function __construct() {
        $this->alertModel = new Alert();
        $this->mailer = new SimpleMailer(SMTP_USER, SITE_NAME);
    }

    // Send Email using SimpleMailer
    public function sendEmail($to, $subject, $body) {
        try {
            // Check if email is configured
            if (!defined('SMTP_USER') || SMTP_USER === 'your_email@gmail.com') {
                // Email not configured, skip silently
                return false;
            }
            
            $htmlBody = $this->getEmailTemplate($subject, $body);
            $result = @$this->mailer->send($to, $subject, $htmlBody);
            
            if ($result) {
                // Log alert with user_id
                $this->alertModel->logAlert($subject, "Email sent to: $to", 'email', null);
            }
            
            return $result;
        } catch (Exception $e) {
            error_log("Email Error: " . $e->getMessage());
            return false;
        }
    }

    // Send Telegram Message
    public function sendTelegram($message) {
        try {
            // Check if Telegram is configured
            if (!defined('TELEGRAM_BOT_TOKEN') || 
                !defined('TELEGRAM_CHAT_ID') || 
                TELEGRAM_BOT_TOKEN === 'your_telegram_bot_token' || 
                TELEGRAM_CHAT_ID === 'your_telegram_chat_id') {
                // Telegram not configured, skip silently
                return false;
            }
            
            $url = "https://api.telegram.org/bot" . TELEGRAM_BOT_TOKEN . "/sendMessage";
            $data = [
                'chat_id' => TELEGRAM_CHAT_ID,
                'text' => $message,
                'parse_mode' => 'HTML'
            ];

            $options = [
                'http' => [
                    'method' => 'POST',
                    'header' => 'Content-Type: application/x-www-form-urlencoded',
                    'content' => http_build_query($data),
                    'ignore_errors' => true
                ]
            ];

            $context = stream_context_create($options);
            $result = @file_get_contents($url, false, $context);
            
            if ($result !== false) {
                // Log alert only if successful
                $this->alertModel->logAlert('Telegram Alert', $message, 'telegram', null);
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            error_log("Telegram Error: " . $e->getMessage());
            return false;
        }
    }

    // Send Telegram to User's Number
    public function sendUserTelegram($userTelegramNumber, $message) {
        try {
            error_log("sendUserTelegram called with number: $userTelegramNumber");
            
            // Check if user has telegram number
            if (empty($userTelegramNumber)) {
                error_log("User telegram number is empty");
                return false;
            }
            
            // Check if Telegram bot is configured
            if (!defined('TELEGRAM_BOT_TOKEN') || TELEGRAM_BOT_TOKEN === 'your_telegram_bot_token') {
                error_log("Telegram bot token not configured");
                return false;
            }
            
            $url = "https://api.telegram.org/bot" . TELEGRAM_BOT_TOKEN . "/sendMessage";
            $data = [
                'chat_id' => $userTelegramNumber,
                'text' => $message,
                'parse_mode' => 'HTML'
            ];

            $options = [
                'http' => [
                    'method' => 'POST',
                    'header' => 'Content-Type: application/x-www-form-urlencoded',
                    'content' => http_build_query($data),
                    'ignore_errors' => true
                ]
            ];

            $context = stream_context_create($options);
            $result = @file_get_contents($url, false, $context);
            
            error_log("Telegram API response: " . ($result ?: 'empty'));
            
            if ($result !== false) {
                error_log("User Telegram sent successfully to: $userTelegramNumber");
                return true;
            }
            
            error_log("Telegram API returned false");
            return false;
        } catch (Exception $e) {
            error_log("User Telegram Error: " . $e->getMessage());
            return false;
        }
    }

    // Email Template
    private function getEmailTemplate($subject, $body) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <meta name='viewport' content='width=device-width, initial-scale=1.0'>
            <style>
                body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); margin: 0; padding: 20px; }
                .container { max-width: 600px; margin: 0 auto; background: rgba(255, 255, 255, 0.95); border-radius: 20px; padding: 40px; box-shadow: 0 20px 60px rgba(0,0,0,0.3); }
                .header { text-align: center; margin-bottom: 30px; }
                .header h1 { color: #667eea; margin: 0; font-size: 28px; text-shadow: 2px 2px 4px rgba(0,0,0,0.1); }
                .content { color: #333; line-height: 1.8; font-size: 16px; }
                .footer { text-align: center; margin-top: 30px; padding-top: 20px; border-top: 2px solid #667eea; color: #666; font-size: 14px; }
                .button { display: inline-block; padding: 12px 30px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; text-decoration: none; border-radius: 25px; margin: 20px 0; font-weight: bold; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>🔍 " . SITE_NAME . "</h1>
                </div>
                <div class='content'>
                    <h2 style='color: #764ba2;'>{$subject}</h2>
                    {$body}
                </div>
                <div class='footer'>
                    <p>&copy; " . date('Y') . " " . SITE_NAME . ". All rights reserved.</p>
                    <p>This is an automated email. Please do not reply.</p>
                </div>
            </div>
        </body>
        </html>
        ";
    }

    // Notification Templates
    public function notifyNewUser($user) {
        $subject = "Welcome to " . SITE_NAME;
        $body = "
            <p>Hello <strong>{$user['name']}</strong>,</p>
            <p>Welcome to our platform! Your account has been successfully created.</p>
            <p><strong>Email:</strong> {$user['email']}</p>
            <p>You can now log in and explore our services.</p>
            <a href='" . SITE_URL . "/public/login.php' class='button'>Login Now</a>
        ";
        $this->sendEmail($user['email'], $subject, $body);
        
        $telegramMsg = "🆕 <b>New User Registered</b>\n\n";
        $telegramMsg .= "👤 Name: {$user['name']}\n";
        $telegramMsg .= "📧 Email: {$user['email']}\n";
        $telegramMsg .= "🕐 Time: " . date('Y-m-d H:i:s');
        $this->sendTelegram($telegramMsg);
    }

    public function notifyPaymentSuccess($user, $plan, $payment) {
        $subject = "Payment Successful - " . $plan['name'];
        $body = "
            <p>Hello <strong>{$user['name']}</strong>,</p>
            <p>Your payment has been successfully processed!</p>
            <p><strong>Plan:</strong> {$plan['name']}</p>
            <p><strong>Amount:</strong> ₹{$payment['amount']}</p>
            <p><strong>Payment ID:</strong> {$payment['razorpay_payment_id']}</p>
            <p>Your subscription is now active. Enjoy our services!</p>
            <a href='" . SITE_URL . "/public/dashboard.php' class='button'>Go to Dashboard</a>
        ";
        $this->sendEmail($user['email'], $subject, $body);
        
        // Send to admin
        $telegramMsg = "💰 <b>Payment Success</b>\n\n";
        $telegramMsg .= "👤 User: {$user['name']}\n";
        $telegramMsg .= "📦 Plan: {$plan['name']}\n";
        $telegramMsg .= "💵 Amount: ₹{$payment['amount']}\n";
        $telegramMsg .= "🆔 Payment ID: {$payment['razorpay_payment_id']}\n";
        $telegramMsg .= "🕐 Time: " . date('Y-m-d H:i:s');
        $this->sendTelegram($telegramMsg);
        
        // Send to user if they have Telegram
        if (!empty($user['telegram_number'])) {
            $userMsg = "🎉 <b>Payment Successful!</b>\n\n";
            $userMsg .= "✅ Your {$plan['name']} subscription has been activated!\n\n";
            $userMsg .= "💰 Amount Paid: ₹{$payment['amount']}\n";
            $userMsg .= "📦 Plan: {$plan['name']}\n";
            $userMsg .= "🆔 Payment ID: {$payment['razorpay_payment_id']}\n\n";
            $userMsg .= "Thank you for your purchase! 🙏";
            $this->sendUserTelegram($user['telegram_number'], $userMsg);
            // Log user notification
            $this->alertModel->logAlert('Payment Success', $userMsg, 'telegram', $user['id']);
        }
    }

    public function notifyNewSearch($user, $searchType, $queryValue, $source) {
        $subject = "New Search Performed - " . ucfirst($searchType);
        $body = "
            <p>Hello <strong>{$user['name']}</strong>,</p>
            <p>Your search has been completed successfully!</p>
            <p><strong>Search Type:</strong> " . ucfirst($searchType) . "</p>
            <p><strong>Query:</strong> {$queryValue}</p>
            <p><strong>Source:</strong> " . ucfirst($source) . "</p>
            <p>View your search history in the dashboard.</p>
            <a href='" . SITE_URL . "/public/history.php' class='button'>View History</a>
        ";
        $this->sendEmail($user['email'], $subject, $body);
        
        // Send to admin
        $telegramMsg = "🔍 <b>New Search</b>\n\n";
        $telegramMsg .= "👤 User: {$user['name']}\n";
        $telegramMsg .= "🔎 Type: " . ucfirst($searchType) . "\n";
        $telegramMsg .= "📱 Query: {$queryValue}\n";
        $telegramMsg .= "📊 Source: " . ucfirst($source) . "\n";
        $telegramMsg .= "🕐 Time: " . date('Y-m-d H:i:s');
        $this->sendTelegram($telegramMsg);
        
        // Send to user if they have Telegram
        if (!empty($user['telegram_number'])) {
            $userMsg = "🔍 <b>Search Completed!</b>\n\n";
            $userMsg .= "✅ Your " . ucfirst($searchType) . " search is ready!\n\n";
            $userMsg .= "📱 Query: {$queryValue}\n";
            $userMsg .= "📊 Source: " . ucfirst($source) . "\n\n";
            $userMsg .= "View results in your dashboard 👉";
            $this->sendUserTelegram($user['telegram_number'], $userMsg);
            // Log user notification
            $this->alertModel->logAlert('Search Completed', $userMsg, 'telegram', $user['id']);
        }
    }

    public function notifyAPIFailure($searchType, $queryValue, $error) {
        $telegramMsg = "⚠️ <b>API Failure</b>\n\n";
        $telegramMsg .= "🔎 Type: " . ucfirst($searchType) . "\n";
        $telegramMsg .= "📱 Query: {$queryValue}\n";
        $telegramMsg .= "❌ Error: {$error}\n";
        $telegramMsg .= "🕐 Time: " . date('Y-m-d H:i:s');
        $this->sendTelegram($telegramMsg);
    }

    public function notifySubscriptionActivated($user, $plan, $subscription) {
        $subject = "Subscription Activated - " . $plan['name'];
        $body = "
            <p>Hello <strong>{$user['name']}</strong>,</p>
            <p>Your subscription has been activated!</p>
            <p><strong>Plan:</strong> {$plan['name']}</p>
            <p><strong>Search Limit:</strong> {$subscription['total_search_limit']} searches</p>
            <p><strong>Valid Until:</strong> " . date('d M Y', strtotime($subscription['end_date'])) . "</p>
            <p>Start exploring now!</p>
            <a href='" . SITE_URL . "/public/dashboard.php' class='button'>Go to Dashboard</a>
        ";
        $this->sendEmail($user['email'], $subject, $body);
        
        $telegramMsg = "✅ <b>Subscription Activated</b>\n\n";
        $telegramMsg .= "👤 User: {$user['name']}\n";
        $telegramMsg .= "📦 Plan: {$plan['name']}\n";
        $telegramMsg .= "🔢 Searches: {$subscription['total_search_limit']}\n";
        $telegramMsg .= "📅 Valid Until: " . date('d M Y', strtotime($subscription['end_date'])) . "\n";
        $telegramMsg .= "🕐 Time: " . date('Y-m-d H:i:s');
        $this->sendTelegram($telegramMsg);
    }
}
