<?php
/**
 * Simple Email Sender with Direct SMTP Support (No Composer Required)
 * Uses fsockopen to connect directly to SMTP server
 */

class SimpleMailer {
    private $from;
    private $fromName;
    private $smtpHost;
    private $smtpPort;
    private $smtpUser;
    private $smtpPass;
    
    public function __construct($from, $fromName = '') {
        $this->from = $from;
        $this->fromName = $fromName ?: $from;
        
        // Load SMTP config from constants
        $this->smtpHost = defined('SMTP_HOST') ? SMTP_HOST : 'smtp.gmail.com';
        $this->smtpPort = defined('SMTP_PORT') ? SMTP_PORT : 587;
        $this->smtpUser = defined('SMTP_USER') ? SMTP_USER : $from;
        $this->smtpPass = defined('SMTP_PASS') ? SMTP_PASS : '';
    }
    
    /**
     * Send HTML Email using SMTP
     */
    public function send($to, $subject, $htmlBody) {
        try {
            // Try SMTP first
            return $this->sendViaSMTP($to, $subject, $htmlBody);
        } catch (Exception $e) {
            error_log("SMTP Error: " . $e->getMessage());
            // Fallback to mail() if SMTP fails
            return $this->sendViaMail($to, $subject, $htmlBody);
        }
    }
    
    /**
     * Send via SMTP using fsockopen
     */
    private function sendViaSMTP($to, $subject, $htmlBody) {
        // Connect to SMTP server
        $smtp = @fsockopen('tls://' . $this->smtpHost, $this->smtpPort, $errno, $errstr, 30);
        
        if (!$smtp) {
            throw new Exception("Failed to connect to SMTP server: $errstr ($errno)");
        }
        
        // Read server response
        $response = fgets($smtp, 515);
        if (substr($response, 0, 3) != '220') {
            throw new Exception("SMTP Error: $response");
        }
        
        // Send EHLO
        fputs($smtp, "EHLO " . $this->smtpHost . "\r\n");
        $response = fgets($smtp, 515);
        
        // AUTH LOGIN
        fputs($smtp, "AUTH LOGIN\r\n");
        $response = fgets($smtp, 515);
        
        fputs($smtp, base64_encode($this->smtpUser) . "\r\n");
        $response = fgets($smtp, 515);
        
        fputs($smtp, base64_encode($this->smtpPass) . "\r\n");
        $response = fgets($smtp, 515);
        
        if (substr($response, 0, 3) != '235') {
            fclose($smtp);
            throw new Exception("SMTP Authentication failed: $response");
        }
        
        // MAIL FROM
        fputs($smtp, "MAIL FROM: <{$this->from}>\r\n");
        $response = fgets($smtp, 515);
        
        // RCPT TO
        fputs($smtp, "RCPT TO: <$to>\r\n");
        $response = fgets($smtp, 515);
        
        // DATA
        fputs($smtp, "DATA\r\n");
        $response = fgets($smtp, 515);
        
        // Email headers and body
        $headers = "From: {$this->fromName} <{$this->from}>\r\n";
        $headers .= "To: $to\r\n";
        $headers .= "Subject: $subject\r\n";
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        $headers .= "\r\n";
        
        fputs($smtp, $headers . $htmlBody . "\r\n.\r\n");
        $response = fgets($smtp, 515);
        
        // QUIT
        fputs($smtp, "QUIT\r\n");
        fclose($smtp);
        
        return true;
    }
    
    /**
     * Fallback: Send using PHP's mail() function
     */
    private function sendViaMail($to, $subject, $htmlBody) {
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: {$this->fromName} <{$this->from}>" . "\r\n";
        $headers .= "Reply-To: {$this->from}" . "\r\n";
        $headers .= "X-Mailer: PHP/" . phpversion();
        
        return @mail($to, $subject, $htmlBody, $headers);
    }
}
