<?php
/**
 * Simple Razorpay Integration (No Composer Required)
 * Direct API calls using cURL
 */

class SimpleRazorpay {
    private $keyId;
    private $keySecret;
    
    public function __construct($keyId, $keySecret) {
        $this->keyId = $keyId;
        $this->keySecret = $keySecret;
    }
    
    /**
     * Create Razorpay Order
     */
    public function createOrder($amount, $currency = 'INR', $receipt = null) {
        $url = 'https://api.razorpay.com/v1/orders';
        
        $data = [
            'amount' => $amount * 100, // Convert to paise
            'currency' => $currency,
            'receipt' => $receipt ?? 'rcpt_' . time(),
            'payment_capture' => 1
        ];
        
        $response = $this->makeRequest($url, 'POST', $data);
        return $response;
    }
    
    /**
     * Verify Payment Signature
     */
    public function verifySignature($orderId, $paymentId, $signature) {
        $text = $orderId . '|' . $paymentId;
        $generatedSignature = hash_hmac('sha256', $text, $this->keySecret);
        
        return hash_equals($generatedSignature, $signature);
    }
    
    /**
     * Fetch Payment Details
     */
    public function fetchPayment($paymentId) {
        $url = 'https://api.razorpay.com/v1/payments/' . $paymentId;
        $response = $this->makeRequest($url, 'GET');
        return $response;
    }
    
    /**
     * Make cURL Request
     */
    private function makeRequest($url, $method = 'GET', $data = null) {
        $ch = curl_init();
        
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_USERPWD, $this->keyId . ':' . $this->keySecret);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json'
        ]);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode >= 200 && $httpCode < 300) {
            return json_decode($response, true);
        } else {
            return [
                'error' => true,
                'message' => 'API request failed',
                'response' => json_decode($response, true)
            ];
        }
    }
}
