<?php
require_once __DIR__ . '/Database.php';

class AIChatbot {
    private $db;
    private $settings;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
        $this->loadSettings();
    }
    
    private function loadSettings() {
        $stmt = $this->db->query("SELECT * FROM chatbot_settings ORDER BY id DESC LIMIT 1");
        $this->settings = $stmt->fetch();
        
        if (!$this->settings) {
            // Create default settings if not exists
            $this->db->exec("INSERT INTO chatbot_settings (is_enabled) VALUES (TRUE)");
            $this->loadSettings();
        }
    }
    
    public function isEnabled() {
        return $this->settings['is_enabled'] == 1;
    }
    
    public function shouldAutoReply($userId) {
        if (!$this->isEnabled()) {
            return false;
        }
        
        // Check if admin has replied recently
        $delay = $this->settings['auto_reply_delay'];
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as count 
            FROM chat_messages 
            WHERE user_id = ? 
            AND sender_type = 'admin' 
            AND is_bot = FALSE
            AND created_at > DATE_SUB(NOW(), INTERVAL ? SECOND)
        ");
        $stmt->execute([$userId, $delay]);
        $result = $stmt->fetch();
        
        return $result['count'] == 0;
    }
    
    public function generateResponse($userId, $userMessage) {
        $startTime = microtime(true);
        
        // Try quick response first (with user context)
        $quickResponse = $this->getQuickResponse($userMessage, $userId);
        if ($quickResponse) {
            $processingTime = (microtime(true) - $startTime) * 1000;
            $this->logConversation($userId, null, $userMessage, $quickResponse, 'quick', 1.0, $processingTime);
            return $quickResponse;
        }
        
        // Try AI response
        if ($this->settings['ai_provider'] === 'gemini' && !empty($this->settings['api_key'])) {
            $aiResponse = $this->getGeminiResponse($userId, $userMessage);
            if ($aiResponse) {
                $processingTime = (microtime(true) - $startTime) * 1000;
                $this->logConversation($userId, null, $userMessage, $aiResponse, 'ai', 0.8, $processingTime);
                return $aiResponse;
            }
        }
        
        // Fallback response with user's name
        $userContext = $this->getUserContext($userId);
        $userName = $userContext['name'];
        $fallback = "Hello {$userName}! 👋 Welcome to Gravity. How can I help you today?";
        $processingTime = (microtime(true) - $startTime) * 1000;
        $this->logConversation($userId, null, $userMessage, $fallback, 'fallback', 0.5, $processingTime);
        
        return $fallback;
    }
    
    private function getQuickResponse($message, $userId = null) {
        $message = strtolower(trim($message));
        
        // Check for user-specific queries first
        if ($userId) {
            $userContext = $this->getUserContext($userId);
            
            // Plan status query
            if (preg_match('/(my plan|subscription status|plan status|my subscription)/i', $message)) {
                $status = "📊 **Your Subscription Details:**\n\n";
                $status .= "✅ Plan: {$userContext['plan_name']}\n";
                $status .= "📈 Status: " . ucfirst($userContext['plan_status']) . "\n";
                $status .= "🔍 Searches: {$userContext['searches_used']}/{$userContext['search_limit']}\n";
                $status .= "⏰ Expires: {$userContext['expires_at']}\n\n";
                
                if ($userContext['searches_remaining'] <= 0) {
                    $status .= "⚠️ You've used all your searches! Please upgrade your plan.";
                } elseif ($userContext['searches_remaining'] < 5) {
                    $status .= "⚠️ Only {$userContext['searches_remaining']} searches remaining!";
                } else {
                    $status .= "✨ You have {$userContext['searches_remaining']} searches remaining!";
                }
                
                return $status;
            }
            
            // Searches remaining query
            if (preg_match('/(searches left|remaining searches|how many searches)/i', $message)) {
                $remaining = $userContext['searches_remaining'];
                $total = $userContext['search_limit'];
                $used = $userContext['searches_used'];
                
                $response = "🔍 **Search Usage:**\n\n";
                $response .= "Used: $used\n";
                $response .= "Remaining: $remaining\n";
                $response .= "Total Limit: $total\n\n";
                
                if ($remaining > 10) {
                    $response .= "✅ You're good to go!";
                } elseif ($remaining > 0) {
                    $response .= "⚠️ Running low! Consider upgrading.";
                } else {
                    $response .= "❌ No searches left! Please upgrade your plan.";
                }
                
                return $response;
            }
            
            // Expiry query
            if (preg_match('/(expire|expiry|valid till|plan expire)/i', $message)) {
                $response = "📅 **Plan Expiry:**\n\n";
                $response .= "Your {$userContext['plan_name']} plan ";
                
                if ($userContext['expires_at'] === 'N/A') {
                    $response .= "is not active.\n\n";
                    $response .= "💡 Purchase a plan to start searching!";
                } else {
                    $response .= "expires on **{$userContext['expires_at']}**\n\n";
                    
                    $daysLeft = (strtotime($userContext['expires_at']) - time()) / (60 * 60 * 24);
                    if ($daysLeft < 7) {
                        $response .= "⚠️ Expiring soon! Renew to avoid interruption.";
                    } else {
                        $response .= "✅ Your plan is active!";
                    }
                }
                
                return $response;
            }
            
            
            // Recent searches query - multiple patterns for flexibility
            if (preg_match('/\b(show|view|see|get|display)\s+(my\s+)?(recent|last|previous)\s+(\d+\s+)?searches?\b/i', $message) ||
                preg_match('/\b(recent|last|previous)\s+(\d+\s+)?searches?\b/i', $message) ||
                preg_match('/\bsearch\s+(history|log)\b/i', $message) ||
                preg_match('/\bmy\s+searches?\b/i', $message)) {
                $searches = $this->getRecentSearches($userId, 5);
                
                $response = "🔍 **Your Recent Searches:**\n\n";
                
                if (empty($searches)) {
                    $response .= "No searches found yet.\n\n";
                    $response .= "💡 Start searching for mobile numbers or Aadhaar details!";
                } else {
                    foreach ($searches as $index => $search) {
                        $num = $index + 1;
                        $type = ucfirst($search['search_type']);
                        $query = substr($search['query_value'], 0, 20);
                        $time = date('d M, h:i A', strtotime($search['created_at']));
                        $response .= "$num. $type: $query\n   📅 $time\n\n";
                    }
                }
                
                return $response;
            }
            
            // Profile details query - more specific
            if (preg_match('/\b(show|view|see|get|display)\s+(my\s+)?profile\b/i', $message) ||
                preg_match('/\baccount\s+(details|info|information)\b/i', $message) ||
                preg_match('/\bmy\s+(account|info|information)\b/i', $message)) {
                $response = "👤 **Your Profile:**\n\n";
                $response .= "📛 Name: {$userContext['name']}\n";
                $response .= "📧 Email: {$userContext['email']}\n";
                $response .= "📅 Member Since: {$userContext['member_since']}\n";
                $response .= "📊 Current Plan: {$userContext['plan_name']}\n";
                $response .= "🔍 Total Searches: {$userContext['searches_used']}\n\n";
                $response .= "✨ Need to update your profile? Visit the dashboard!";
                
                return $response;
            }
        }
        
        // Standard quick responses
        $stmt = $this->db->query("
            SELECT * FROM chatbot_quick_responses 
            WHERE is_active = TRUE 
            ORDER BY priority DESC
        ");
        $responses = $stmt->fetchAll();
        
        foreach ($responses as $response) {
            $keywords = explode(',', strtolower($response['keywords']));
            foreach ($keywords as $keyword) {
                $keyword = trim($keyword);
                if (strpos($message, $keyword) !== false) {
                    // Personalize response with user's name if {name} placeholder exists
                    $personalizedResponse = str_replace('{name}', $userContext['name'], $response['response']);
                    return $personalizedResponse;
                }
            }
        }
        
        return null;
    }
    
    private function getGeminiResponse($userId, $userMessage) {
        $apiKey = $this->settings['api_key'];
        if (empty($apiKey)) {
            return null;
        }
        
        // Get user context (subscription, usage, etc.)
        $userContext = $this->getUserContext($userId);
        
        // Get conversation context (last 5 messages)
        $context = $this->getConversationContext($userId, 5);
        
        // Build prompt with context
        $systemPrompt = $this->settings['system_prompt'];
        
        // Add user-specific context to prompt
        $userInfo = "\n\nUser Information (Always address the user by their name in your responses):\n";
        $userInfo .= "- Name: {$userContext['name']}\n";
        $userInfo .= "- Email: {$userContext['email']}\n";
        $userInfo .= "- Subscription: {$userContext['plan_name']}\n";
        $userInfo .= "- Plan Status: {$userContext['plan_status']}\n";
        $userInfo .= "- Searches Used: {$userContext['searches_used']}/{$userContext['search_limit']}\n";
        $userInfo .= "- Searches Remaining: {$userContext['searches_remaining']}\n";
        $userInfo .= "- Plan Expires: {$userContext['expires_at']}\n";
        
        $conversationHistory = "";
        
        foreach ($context as $msg) {
            $role = $msg['sender_type'] === 'user' ? 'User' : 'Assistant';
            $conversationHistory .= "$role: {$msg['message']}\n";
        }
        
        $fullPrompt = $systemPrompt . $userInfo . "\n\nConversation History:\n" . $conversationHistory . "\nUser: " . $userMessage . "\nAssistant:";
        
        // Call Gemini API
        $url = "https://generativelanguage.googleapis.com/v1beta/models/gemini-pro:generateContent?key=" . $apiKey;
        
        $data = [
            'contents' => [
                [
                    'parts' => [
                        ['text' => $fullPrompt]
                    ]
                ]
            ],
            'generationConfig' => [
                'temperature' => floatval($this->settings['temperature']),
                'maxOutputTokens' => intval($this->settings['max_tokens'])
            ]
        ];
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200 && $response) {
            $result = json_decode($response, true);
            if (isset($result['candidates'][0]['content']['parts'][0]['text'])) {
                return trim($result['candidates'][0]['content']['parts'][0]['text']);
            }
        }
        
        error_log("Gemini API Error: HTTP $httpCode - " . $response);
        return null;
    }
    
    private function getUserContext($userId) {
        $stmt = $this->db->prepare("
            SELECT 
                u.name,
                u.email,
                u.role,
                u.created_at as member_since,
                p.name as plan_name,
                s.status as plan_status,
                s.total_search_limit as search_limit,
                s.used_searches as searches_used,
                (s.total_search_limit - s.used_searches) as searches_remaining,
                s.end_date as expires_at
            FROM users u
            LEFT JOIN subscriptions s ON u.id = s.user_id AND s.status = 'active'
            LEFT JOIN plans p ON s.plan_id = p.id
            WHERE u.id = ?
            ORDER BY s.total_search_limit DESC, s.end_date DESC
            LIMIT 1
        ");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return [
                'name' => 'User',
                'email' => 'unknown',
                'plan_name' => 'No Plan',
                'plan_status' => 'inactive',
                'search_limit' => 0,
                'searches_used' => 0,
                'searches_remaining' => 0,
                'expires_at' => 'N/A'
            ];
        }
        
        // Format the data
        return [
            'name' => $user['name'] ?? 'User',
            'email' => $user['email'] ?? 'unknown',
            'plan_name' => $user['plan_name'] ?? 'No Active Plan',
            'plan_status' => $user['plan_status'] ?? 'inactive',
            'search_limit' => $user['search_limit'] ?? 0,
            'searches_used' => $user['searches_used'] ?? 0,
            'searches_remaining' => max(0, ($user['search_limit'] ?? 0) - ($user['searches_used'] ?? 0)),
            'expires_at' => $user['expires_at'] ? date('d M Y', strtotime($user['expires_at'])) : 'N/A',
            'member_since' => $user['member_since'] ? date('d M Y', strtotime($user['member_since'])) : 'N/A'
        ];
    }
    
    private function getRecentSearches($userId, $limit = 5) {
        $stmt = $this->db->prepare("
            SELECT search_type, query_value, created_at
            FROM searches
            WHERE user_id = ?
            ORDER BY created_at DESC
            LIMIT ?
        ");
        $stmt->execute([$userId, $limit]);
        return $stmt->fetchAll();
    }
    
    private function getConversationContext($userId, $limit = 5) {
        $stmt = $this->db->prepare("
            SELECT sender_type, message, created_at 
            FROM chat_messages 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT ?
        ");
        $stmt->execute([$userId, $limit]);
        return array_reverse($stmt->fetchAll());
    }
    
    public function sendBotMessage($userId, $message) {
        $stmt = $this->db->prepare("
            INSERT INTO chat_messages (user_id, sender_type, is_bot, message) 
            VALUES (?, 'admin', TRUE, ?)
        ");
        return $stmt->execute([$userId, $message]);
    }
    
    private function logConversation($userId, $messageId, $userMessage, $botResponse, $type, $confidence, $processingTime) {
        $stmt = $this->db->prepare("
            INSERT INTO chatbot_logs 
            (user_id, message_id, user_message, bot_response, response_type, confidence_score, processing_time_ms) 
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $userId, 
            $messageId, 
            $userMessage, 
            $botResponse, 
            $type, 
            $confidence, 
            round($processingTime)
        ]);
    }
    
    // Admin functions
    public function updateSettings($data) {
        $stmt = $this->db->prepare("
            UPDATE chatbot_settings SET 
            is_enabled = ?,
            auto_reply_delay = ?,
            ai_provider = ?,
            api_key = ?,
            system_prompt = ?,
            max_tokens = ?,
            temperature = ?
            WHERE id = ?
        ");
        return $stmt->execute([
            $data['is_enabled'],
            $data['auto_reply_delay'],
            $data['ai_provider'],
            $data['api_key'],
            $data['system_prompt'],
            $data['max_tokens'],
            $data['temperature'],
            $this->settings['id']
        ]);
    }
    
    public function getSettings() {
        return $this->settings;
    }
    
    public function getStats($days = 7) {
        $stmt = $this->db->prepare("
            SELECT 
                COUNT(*) as total_conversations,
                AVG(confidence_score) as avg_confidence,
                AVG(processing_time_ms) as avg_response_time,
                SUM(CASE WHEN response_type = 'quick' THEN 1 ELSE 0 END) as quick_responses,
                SUM(CASE WHEN response_type = 'ai' THEN 1 ELSE 0 END) as ai_responses,
                SUM(CASE WHEN response_type = 'fallback' THEN 1 ELSE 0 END) as fallback_responses
            FROM chatbot_logs
            WHERE created_at > DATE_SUB(NOW(), INTERVAL ? DAY)
        ");
        $stmt->execute([$days]);
        return $stmt->fetch();
    }
}
