<?php
class ActivityLog {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Log user activity
     */
    public function logActivity($userId, $action, $description = '', $data = []) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO activity_logs 
                (user_id, action, description, ip_address, user_agent, data) 
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            
            $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
            $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
            $jsonData = json_encode($data);
            
            return $stmt->execute([
                $userId,
                $action,
                $description,
                $ipAddress,
                $userAgent,
                $jsonData
            ]);
        } catch (Exception $e) {
            error_log("Activity log error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get user activity logs
     */
    public function getUserActivities($userId, $limit = 50, $offset = 0) {
        try {
            $stmt = $this->db->prepare("
                SELECT * FROM activity_logs 
                WHERE user_id = ? 
                ORDER BY created_at DESC 
                LIMIT ? OFFSET ?
            ");
            
            $stmt->execute([$userId, $limit, $offset]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Get activities error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get activity count by type
     */
    public function getActivityStats($userId) {
        try {
            $stmt = $this->db->prepare("
                SELECT 
                    action,
                    COUNT(*) as count,
                    MAX(created_at) as last_activity
                FROM activity_logs 
                WHERE user_id = ? 
                GROUP BY action
                ORDER BY count DESC
            ");
            
            $stmt->execute([$userId]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Get activity stats error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get recent activities (last 24 hours)
     */
    public function getRecentActivities($userId, $hours = 24) {
        try {
            $stmt = $this->db->prepare("
                SELECT * FROM activity_logs 
                WHERE user_id = ? 
                AND created_at >= DATE_SUB(NOW(), INTERVAL ? HOUR)
                ORDER BY created_at DESC
            ");
            
            $stmt->execute([$userId, $hours]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Get recent activities error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Delete old activity logs (cleanup)
     */
    public function cleanupOldLogs($days = 90) {
        try {
            $stmt = $this->db->prepare("
                DELETE FROM activity_logs 
                WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
            ");
            
            $stmt->execute([$days]);
            return $stmt->rowCount();
        } catch (Exception $e) {
            error_log("Cleanup logs error: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Get activity icon
     */
    public static function getActivityIcon($action) {
        $icons = [
            'login' => '🔐',
            'logout' => '🚪',
            'search_mobile' => '📱',
            'search_aadhaar' => '🆔',
            'plan_purchase' => '💳',
            'profile_update' => '✏️',
            'password_change' => '🔒',
            'theme_change' => '🎨',
            'export_data' => '📥',
            'view_search' => '👁️',
            'failed_login' => '❌',
            'session_expired' => '⏰'
        ];
        
        return $icons[$action] ?? '📝';
    }
    
    /**
     * Get activity label
     */
    public static function getActivityLabel($action) {
        $labels = [
            'login' => 'Logged In',
            'logout' => 'Logged Out',
            'search_mobile' => 'Mobile Search',
            'search_aadhaar' => 'Aadhaar Search',
            'plan_purchase' => 'Plan Purchase',
            'profile_update' => 'Profile Updated',
            'password_change' => 'Password Changed',
            'theme_change' => 'Theme Changed',
            'export_data' => 'Data Exported',
            'view_search' => 'Viewed Search',
            'failed_login' => 'Failed Login',
            'session_expired' => 'Session Expired'
        ];
        
        return $labels[$action] ?? ucwords(str_replace('_', ' ', $action));
    }
}
