<?php
require_once __DIR__ . '/Database.php';

class Chat {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    // Send message
    public function sendMessage($userId, $senderType, $message) {
        $stmt = $this->db->prepare("INSERT INTO chat_messages (user_id, sender_type, message) VALUES (?, ?, ?)");
        return $stmt->execute([$userId, $senderType, $message]);
    }

    // Get user messages
    public function getUserMessages($userId, $limit = 50) {
        $stmt = $this->db->prepare("
            SELECT cm.*, u.name as user_name, u.email 
            FROM chat_messages cm
            JOIN users u ON cm.user_id = u.id
            WHERE cm.user_id = ?
            ORDER BY cm.created_at ASC
            LIMIT ?
        ");
        $stmt->execute([$userId, $limit]);
        return $stmt->fetchAll();
    }

    // Get all chats (for admin)
    public function getAllChats() {
        $stmt = $this->db->query("
            SELECT u.id as user_id, u.name, u.email,
                   COUNT(cm.id) as message_count,
                   MAX(cm.created_at) as last_message_time,
                   SUM(CASE WHEN cm.is_read = 0 AND cm.sender_type = 'user' THEN 1 ELSE 0 END) as unread_count
            FROM users u
            LEFT JOIN chat_messages cm ON u.id = cm.user_id
            WHERE u.role = 'user'
            GROUP BY u.id, u.name, u.email
            HAVING message_count > 0
            ORDER BY last_message_time DESC
        ");
        return $stmt->fetchAll();
    }

    // Mark messages as read
    public function markAsRead($userId, $senderType) {
        $stmt = $this->db->prepare("UPDATE chat_messages SET is_read = 1 WHERE user_id = ? AND sender_type = ?");
        return $stmt->execute([$userId, $senderType]);
    }

    // Get unread count
    public function getUnreadCount($userId, $senderType) {
        $stmt = $this->db->prepare("SELECT COUNT(*) as count FROM chat_messages WHERE user_id = ? AND sender_type = ? AND is_read = 0");
        $stmt->execute([$userId, $senderType]);
        return $stmt->fetch()['count'];
    }

    // Get new messages (for polling)
    public function getNewMessages($userId, $lastMessageId = 0) {
        $stmt = $this->db->prepare("
            SELECT cm.*, u.name as user_name 
            FROM chat_messages cm
            JOIN users u ON cm.user_id = u.id
            WHERE cm.user_id = ? AND cm.id > ?
            ORDER BY cm.created_at ASC
        ");
        $stmt->execute([$userId, $lastMessageId]);
        return $stmt->fetchAll();
    }
}
