<?php
require_once __DIR__ . '/Database.php';

class Search {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function getCachedSearch($searchType, $queryValue) {
        $stmt = $this->db->prepare("SELECT * FROM searches WHERE search_type = ? AND query_value = ?");
        $stmt->execute([$searchType, $queryValue]);
        return $stmt->fetch();
    }

    public function saveSearch($userId, $searchType, $queryValue, $jsonResponse, $source = 'api') {
        // Always insert new search (allows multiple searches for same number)
        $stmt = $this->db->prepare("
            INSERT INTO searches (user_id, search_type, query_value, json_response, source) 
            VALUES (?, ?, ?, ?, ?)
        ");
        return $stmt->execute([$userId, $searchType, $queryValue, $jsonResponse, $source]);
    }

    public function getSearchById($searchId) {
        $stmt = $this->db->prepare("SELECT * FROM searches WHERE id = ?");
        $stmt->execute([$searchId]);
        return $stmt->fetch();
    }

    public function getUserSearches($userId, $limit = 50) {
        $stmt = $this->db->prepare("SELECT * FROM searches WHERE user_id = ? ORDER BY created_at DESC LIMIT ?");
        $stmt->execute([$userId, $limit]);
        return $stmt->fetchAll();
    }

    public function getAllSearches($limit = 100) {
        $stmt = $this->db->prepare("
            SELECT s.*, u.name as user_name, u.email 
            FROM searches s 
            JOIN users u ON s.user_id = u.id 
            ORDER BY s.created_at DESC 
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        return $stmt->fetchAll();
    }

    public function getTotalSearches() {
        $stmt = $this->db->query("SELECT COUNT(*) as total FROM searches");
        return $stmt->fetch()['total'];
    }

    public function getTodaySearches() {
        $stmt = $this->db->query("SELECT COUNT(*) as total FROM searches WHERE DATE(created_at) = CURDATE()");
        return $stmt->fetch()['total'];
    }

    public function getSearchStats() {
        $stmt = $this->db->query("
            SELECT 
                COUNT(*) as total_searches,
                SUM(CASE WHEN source = 'api' THEN 1 ELSE 0 END) as api_searches,
                SUM(CASE WHEN source = 'cache' THEN 1 ELSE 0 END) as cache_searches,
                SUM(CASE WHEN search_type = 'mobile' THEN 1 ELSE 0 END) as mobile_searches,
                SUM(CASE WHEN search_type = 'aadhaar' THEN 1 ELSE 0 END) as aadhaar_searches
            FROM searches
        ");
        return $stmt->fetch();
    }
}
