<?php
require_once __DIR__ . '/Database.php';

class Subscription {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function createSubscription($userId, $planId, $startDate, $endDate, $totalSearchLimit) {
        $stmt = $this->db->prepare("INSERT INTO subscriptions (user_id, plan_id, start_date, end_date, total_search_limit) VALUES (?, ?, ?, ?, ?)");
        return $stmt->execute([$userId, $planId, $startDate, $endDate, $totalSearchLimit]);
    }

    public function getActiveSubscription($userId) {
        $stmt = $this->db->prepare("
            SELECT s.*, p.name as plan_name, p.description as plan_description 
            FROM subscriptions s 
            JOIN plans p ON s.plan_id = p.id 
            WHERE s.user_id = ? AND s.status = 'active' AND s.end_date > NOW() 
            ORDER BY s.end_date DESC 
            LIMIT 1
        ");
        $stmt->execute([$userId]);
        return $stmt->fetch();
    }

    public function getUserSubscriptions($userId) {
        $stmt = $this->db->prepare("
            SELECT s.*, p.name as plan_name, p.price 
            FROM subscriptions s 
            JOIN plans p ON s.plan_id = p.id 
            WHERE s.user_id = ? 
            ORDER BY s.created_at DESC
        ");
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }

    public function getAllSubscriptions() {
        $stmt = $this->db->query("
            SELECT s.*, u.name as user_name, u.email, p.name as plan_name 
            FROM subscriptions s 
            JOIN users u ON s.user_id = u.id 
            JOIN plans p ON s.plan_id = p.id 
            ORDER BY s.created_at DESC
        ");
        return $stmt->fetchAll();
    }

    public function incrementUsedSearches($subscriptionId) {
        $stmt = $this->db->prepare("UPDATE subscriptions SET used_searches = used_searches + 1 WHERE id = ?");
        return $stmt->execute([$subscriptionId]);
    }

    public function getRemainingSearches($userId) {
        $subscription = $this->getActiveSubscription($userId);
        if ($subscription) {
            return $subscription['total_search_limit'] - $subscription['used_searches'];
        }
        return 0;
    }

    public function expireOldSubscriptions() {
        $stmt = $this->db->query("UPDATE subscriptions SET status = 'expired' WHERE end_date < NOW() AND status = 'active'");
        return $stmt->rowCount();
    }

    public function getTotalActiveSubscriptions() {
        $stmt = $this->db->query("SELECT COUNT(*) as total FROM subscriptions WHERE status = 'active' AND end_date > NOW()");
        return $stmt->fetch()['total'];
    }

    public function assignPlanToUser($userId, $planId) {
        $stmt = $this->db->prepare("SELECT * FROM plans WHERE id = ?");
        $stmt->execute([$planId]);
        $plan = $stmt->fetch();

        if ($plan) {
            $startDate = date('Y-m-d H:i:s');
            $endDate = date('Y-m-d H:i:s', strtotime("+{$plan['duration_days']} days"));
            return $this->createSubscription($userId, $planId, $startDate, $endDate, $plan['search_limit']);
        }
        return false;
    }
}
