<?php
require_once __DIR__ . '/Database.php';

class User
{
    public $db;

    public function __construct()
    {
        $this->db = Database::getInstance()->getConnection();
    }

    public function register($name, $email, $password)
    {
        try {
            $this->db->beginTransaction();

            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $this->db->prepare("INSERT INTO users (name, email, password_hash) VALUES (?, ?, ?)");
            $stmt->execute([$name, $email, $password_hash]);
            $userId = $this->db->lastInsertId();

            // Automatically assign Free Trial subscription
            $this->assignFreeTrial($userId);

            $this->db->commit();
            return $userId;
        } catch (PDOException $e) {
            $this->db->rollBack();
            return false;
        }
    }

    // Assign Free Trial to new user
    public function assignFreeTrial($userId)
    {
        try {
            // Get Free Trial plan (ID = 1, assuming it's the first plan)
            $stmt = $this->db->prepare("SELECT * FROM plans WHERE name = 'Free Trial' AND status = 'active' LIMIT 1");
            $stmt->execute();
            $plan = $stmt->fetch();

            if ($plan) {
                $startDate = date('Y-m-d H:i:s');
                $endDate = date('Y-m-d H:i:s', strtotime('+' . $plan['duration_days'] . ' days'));

                $stmt = $this->db->prepare("
                    INSERT INTO subscriptions 
                    (user_id, plan_id, start_date, end_date, total_search_limit, used_searches, status) 
                    VALUES (?, ?, ?, ?, ?, 0, 'active')
                ");

                return $stmt->execute([
                    $userId,
                    $plan['id'],
                    $startDate,
                    $endDate,
                    $plan['search_limit']
                ]);
            }

            return false;
        } catch (PDOException $e) {
            return false;
        }
    }

    public function login($email, $password)
    {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();

        if ($user && password_verify($password, $user['password_hash'])) {
            if ($user['status'] === 'blocked') {
                return ['error' => 'Your account has been blocked. Please contact administrator.'];
            }

            // Update login info
            $this->updateLoginInfo($user['id'], $_SERVER['REMOTE_ADDR'] ?? 'Unknown', $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown');

            return $user;
        }
        return false;
    }

    public function getUserById($id)
    {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public function getAllUsers()
    {
        $stmt = $this->db->query("SELECT * FROM users ORDER BY created_at DESC");
        return $stmt->fetchAll();
    }

    public function updateUserStatus($userId, $status)
    {
        $stmt = $this->db->prepare("UPDATE users SET status = ? WHERE id = ?");
        return $stmt->execute([$status, $userId]);
    }

    public function updateUserRole($userId, $role)
    {
        $stmt = $this->db->prepare("UPDATE users SET role = ? WHERE id = ?");
        return $stmt->execute([$role, $userId]);
    }

    public function deleteUser($userId)
    {
        $stmt = $this->db->prepare("DELETE FROM users WHERE id = ?");
        return $stmt->execute([$userId]);
    }

    public function getTotalUsers()
    {
        $stmt = $this->db->query("SELECT COUNT(*) as total FROM users");
        return $stmt->fetch()['total'];
    }

    public function getActiveUsers()
    {
        $stmt = $this->db->query("SELECT COUNT(*) as total FROM users WHERE status = 'active'");
        return $stmt->fetch()['total'];
    }

    public function blockUser($userId)
    {
        $stmt = $this->db->prepare("UPDATE users SET status = 'blocked' WHERE id = ?");
        return $stmt->execute([$userId]);
    }

    public function unblockUser($userId)
    {
        $stmt = $this->db->prepare("UPDATE users SET status = 'active' WHERE id = ?");
        return $stmt->execute([$userId]);
    }

    public function makeAdmin($userId)
    {
        $stmt = $this->db->prepare("UPDATE users SET role = 'admin' WHERE id = ?");
        return $stmt->execute([$userId]);
    }

    public function makeUser($userId)
    {
        $stmt = $this->db->prepare("UPDATE users SET role = 'user' WHERE id = ?");
        return $stmt->execute([$userId]);
    }

    // Login tracking
    public function updateLoginInfo($userId, $ip, $device)
    {
        try {
            $stmt = $this->db->prepare("UPDATE users SET last_login = NOW(), last_ip = ?, last_device = ? WHERE id = ?");
            return $stmt->execute([$ip, $device, $userId]);
        } catch (Exception $e) {
            // Silently fail if columns don't exist yet
            return false;
        }
    }

    // Profile update
    public function updateProfile($userId, $name)
    {
        $stmt = $this->db->prepare("UPDATE users SET name = ? WHERE id = ?");
        return $stmt->execute([$name, $userId]);
    }

    public function updatePassword($userId, $newPassword)
    {
        $hash = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $this->db->prepare("UPDATE users SET password_hash = ? WHERE id = ?");
        return $stmt->execute([$hash, $userId]);
    }

    public function verifyPassword($userId, $password)
    {
        $user = $this->getUserById($userId);
        if ($user) {
            return password_verify($password, $user['password_hash']);
        }
        return false;
    }
}
