<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/models/Database.php';
require_once __DIR__ . '/../app/models/ActivityLog.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$activityLog = new ActivityLog();
$activities = [];
$stats = [];
$tableExists = true;

try {
    $activities = $activityLog->getUserActivities($_SESSION['user_id'], 100);
    $stats = $activityLog->getActivityStats($_SESSION['user_id']);
} catch (Exception $e) {
    $tableExists = false;
    error_log("Activity Log Error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Activity Log - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/user-navbar.php'; ?>

    <div class="container mt-4">
        <!-- Header -->
        <div class="glass-card mb-4" data-aos="fade-up">
            <div style="display: flex; align-items: center; gap: 20px; flex-wrap: wrap;">
                <div style="background: var(--primary-gradient); width: 80px; height: 80px; border-radius: 15px; display: flex; align-items: center; justify-content: center; font-size: 36px;">
                    📝
                </div>
                <div style="flex: 1;">
                    <h2 class="gradient-text" style="margin: 0 0 8px 0;">Activity Log</h2>
                    <p style="margin: 0; opacity: 0.8;">Track all your account activities and security events</p>
                </div>
            </div>
        </div>

        <!-- Activity Stats -->
        <?php if (count($stats) > 0): ?>
            <div class="row mb-4">
                <?php foreach (array_slice($stats, 0, 4) as $index => $stat): ?>
                    <div class="col-md-3" data-aos="fade-up" data-aos-delay="<?php echo $index * 100; ?>">
                        <div class="stat-card" style="text-align: center;">
                            <div style="font-size: 32px; margin-bottom: 10px;">
                                <?php echo ActivityLog::getActivityIcon($stat['action']); ?>
                            </div>
                            <h3 class="neon-text"><?php echo $stat['count']; ?></h3>
                            <p style="font-size: 14px;"><?php echo ActivityLog::getActivityLabel($stat['action']); ?></p>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <!-- Activity Timeline -->
        <div class="glass-card" data-aos="fade-up">
            <h3 class="gradient-text mb-4">Recent Activity</h3>
            
            <?php if (count($activities) > 0): ?>
                <div style="max-height: 600px; overflow-y: auto;">
                    <?php foreach ($activities as $activity): 
                        $data = json_decode($activity['data'], true);
                    ?>
                        <div class="activity-item" style="display: flex; gap: 15px; padding: 15px; margin-bottom: 10px; background: rgba(255, 255, 255, 0.03); border-radius: 12px; border-left: 3px solid #667eea;">
                            <div style="font-size: 24px; flex-shrink: 0;">
                                <?php echo ActivityLog::getActivityIcon($activity['action']); ?>
                            </div>
                            <div style="flex: 1; min-width: 0;">
                                <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 5px;">
                                    <strong style="font-size: 15px;">
                                        <?php echo ActivityLog::getActivityLabel($activity['action']); ?>
                                    </strong>
                                    <small style="opacity: 0.7; white-space: nowrap; margin-left: 10px;">
                                        <?php echo date('d M, h:i A', strtotime($activity['created_at'])); ?>
                                    </small>
                                </div>
                                
                                <?php if (!empty($activity['description'])): ?>
                                    <p style="margin: 5px 0; opacity: 0.8; font-size: 14px;">
                                        <?php echo htmlspecialchars($activity['description']); ?>
                                    </p>
                                <?php endif; ?>
                                
                                <div style="display: flex; gap: 15px; margin-top: 8px; font-size: 12px; opacity: 0.6;">
                                    <span>🌐 <?php echo htmlspecialchars($activity['ip_address']); ?></span>
                                    <?php if ($data): ?>
                                        <?php foreach (array_slice($data, 0, 2) as $key => $value): ?>
                                            <span>• <?php echo htmlspecialchars($key); ?>: <?php echo htmlspecialchars(is_array($value) ? json_encode($value) : $value); ?></span>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <div class="text-center" style="padding: 40px;">
                    <?php if (!$tableExists): ?>
                        <div style="font-size: 64px; margin-bottom: 20px;">⚙️</div>
                        <h3 style="color: #ffc107;">Setup Required</h3>
                        <p style="opacity: 0.8; margin: 20px 0;">Activity Log database table needs to be created</p>
                        <div style="background: rgba(255, 193, 7, 0.1); border: 1px solid rgba(255, 193, 7, 0.3); border-radius: 12px; padding: 20px; margin: 20px auto; max-width: 600px; text-align: left;">
                            <strong style="color: #ffc107;">📝 Setup Instructions:</strong>
                            <ol style="margin: 15px 0; padding-left: 20px; opacity: 0.9;">
                                <li style="margin: 10px 0;">Open <strong>phpMyAdmin</strong></li>
                                <li style="margin: 10px 0;">Select your database: <code style="background: rgba(0,0,0,0.3); padding: 2px 8px; border-radius: 4px;">mobile_aadhaar_platform</code></li>
                                <li style="margin: 10px 0;">Click <strong>SQL</strong> tab</li>
                                <li style="margin: 10px 0;">Run the file: <code style="background: rgba(0,0,0,0.3); padding: 2px 8px; border-radius: 4px;">activity_log_system.sql</code></li>
                                <li style="margin: 10px 0;">Refresh this page</li>
                            </ol>
                            <p style="margin: 15px 0 0 0; opacity: 0.8; font-size: 14px;">
                                💡 The SQL file is located in your project root folder
                            </p>
                        </div>
                    <?php else: ?>
                        <div style="font-size: 64px; margin-bottom: 20px; opacity: 0.5;">📝</div>
                        <h3>No Activity Yet</h3>
                        <p style="opacity: 0.7;">Your activity log will appear here after you perform actions</p>
                        <p style="opacity: 0.6; font-size: 14px; margin-top: 15px;">
                            Activities like login, searches, profile updates will be tracked here
                        </p>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <?php include 'includes/chat-widget.php'; ?>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="assets/js/toast.js"></script>
    <script>
        AOS.init({
            duration: 1000,
            once: true
        });
    </script>
</body>
</html>
