<?php
session_start();
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../app/models/Chat.php';
require_once __DIR__ . '/../../app/models/User.php';

// Check if admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit;
}

$chatModel = new Chat();
$userModel = new User();

// Get selected user or first user with messages
$selectedUserId = $_GET['user_id'] ?? 0;
$allChats = $chatModel->getAllChats();

if (!$selectedUserId && count($allChats) > 0) {
    $selectedUserId = $allChats[0]['user_id'];
}

// Get messages for selected user
$messages = [];
$selectedUser = null;
if ($selectedUserId) {
    $messages = $chatModel->getUserMessages($selectedUserId);
    $selectedUser = $userModel->getUserById($selectedUserId);
    // Mark user messages as read
    $chatModel->markAsRead($selectedUserId, 'user');
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Chat - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        .chat-container {
            display: grid;
            grid-template-columns: 300px 1fr;
            gap: 20px;
            height: calc(100vh - 200px);
        }
        
        .chat-users {
            background: var(--glass-bg);
            backdrop-filter: blur(20px);
            border: 1px solid var(--glass-border);
            border-radius: 20px;
            padding: 20px;
            overflow-y: auto;
        }
        
        .chat-user-item {
            padding: 15px;
            margin-bottom: 10px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.3s ease;
            border: 2px solid transparent;
        }
        
        .chat-user-item:hover {
            background: rgba(255, 255, 255, 0.1);
            transform: translateX(5px);
        }
        
        .chat-user-item.active {
            background: var(--primary-gradient);
            border-color: var(--neon-blue);
        }
        
        .chat-user-name {
            font-weight: 600;
            margin-bottom: 5px;
        }
        
        .chat-user-email {
            font-size: 12px;
            opacity: 0.7;
        }
        
        .chat-user-badge {
            background: #f5576c;
            color: white;
            padding: 2px 8px;
            border-radius: 10px;
            font-size: 11px;
            font-weight: bold;
            float: right;
        }
        
        .chat-main {
            background: var(--glass-bg);
            backdrop-filter: blur(20px);
            border: 1px solid var(--glass-border);
            border-radius: 20px;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }
        
        .chat-header {
            padding: 20px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .chat-messages {
            flex: 1;
            overflow-y: auto;
            padding: 20px;
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .chat-message {
            max-width: 70%;
            padding: 12px 16px;
            border-radius: 15px;
            word-wrap: break-word;
            animation: fadeIn 0.3s ease;
        }
        
        .chat-message.user {
            align-self: flex-start;
            background: rgba(255, 255, 255, 0.1);
            border-bottom-left-radius: 5px;
        }
        
        .chat-message.admin {
            align-self: flex-end;
            background: var(--primary-gradient);
            color: white;
            border-bottom-right-radius: 5px;
        }
        
        .chat-message-time {
            font-size: 11px;
            opacity: 0.7;
            margin-top: 5px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .read-status {
            font-size: 10px;
            padding: 2px 6px;
            border-radius: 8px;
            font-weight: 500;
        }
        
        .read-status.seen {
            color: #4ade80;
            background: rgba(74, 222, 128, 0.1);
        }
        
        .read-status.not-seen {
            color: #94a3b8;
            background: rgba(148, 163, 184, 0.1);
        }
        
        .chat-input-area {
            padding: 20px;
            border-top: 1px solid rgba(255, 255, 255, 0.1);
            display: flex;
            gap: 10px;
        }
        
        .chat-input-area input {
            flex: 1;
            padding: 12px 20px;
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 25px;
            color: var(--text-light);
            font-size: 14px;
        }
        
        .chat-input-area input:focus {
            outline: none;
            border-color: var(--neon-blue);
            background: rgba(255, 255, 255, 0.1);
        }
        
        .chat-send-btn {
            padding: 12px 30px;
            background: var(--primary-gradient);
            border: none;
            border-radius: 25px;
            color: white;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .chat-send-btn:hover {
            transform: scale(1.05);
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
        }
        
        .no-chat-selected {
            display: flex;
            align-items: center;
            justify-content: center;
            height: 100%;
            opacity: 0.5;
        }
        
        @media (max-width: 768px) {
            .chat-container {
                grid-template-columns: 1fr;
            }
            .chat-users {
                max-height: 200px;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="container">
            <a href="dashboard.php" class="navbar-brand">🔍 Admin Panel</a>
            <ul class="navbar-menu">
                <li><a href="dashboard.php">Dashboard</a></li>
                <li><a href="users.php">Users</a></li>
                <li><a href="plans.php">Plans</a></li>
                <li><a href="payments.php">Payments</a></li>
                <li><a href="searches.php">Searches</a></li>
                <li><a href="chat.php">Chat</a></li>
                <li><a href="settings.php">Settings</a></li>
                <li><a href="../logout.php" class="btn btn-danger">Logout</a></li>
            </ul>
        </div>
    </nav>

    <div class="container mt-4">
        <h2 class="gradient-text mb-4">💬 User Chats</h2>

        <div class="chat-container">
            <!-- Users List -->
            <div class="chat-users">
                <h4 class="gradient-text mb-3">Active Chats</h4>
                <?php if (count($allChats) > 0): ?>
                    <?php foreach ($allChats as $chat): ?>
                        <a href="?user_id=<?php echo $chat['user_id']; ?>" style="text-decoration: none; color: inherit;">
                            <div class="chat-user-item <?php echo $chat['user_id'] == $selectedUserId ? 'active' : ''; ?>">
                                <?php if ($chat['unread_count'] > 0): ?>
                                    <span class="chat-user-badge"><?php echo $chat['unread_count']; ?></span>
                                <?php endif; ?>
                                <div class="chat-user-name"><?php echo htmlspecialchars($chat['name']); ?></div>
                                <div class="chat-user-email"><?php echo htmlspecialchars($chat['email']); ?></div>
                                <div class="chat-user-email"><?php echo $chat['message_count']; ?> messages</div>
                            </div>
                        </a>
                    <?php endforeach; ?>
                <?php else: ?>
                    <p style="opacity: 0.7; text-align: center;">No chats yet</p>
                <?php endif; ?>
            </div>

            <!-- Chat Area -->
            <div class="chat-main">
                <?php if ($selectedUser): ?>
                    <!-- Chat Header -->
                    <div class="chat-header">
                        <h3 class="gradient-text" style="margin: 0;">
                            💬 Chat with <?php echo htmlspecialchars($selectedUser['name']); ?>
                        </h3>
                        <p style="opacity: 0.7; margin: 5px 0 0 0; font-size: 14px;">
                            <?php echo htmlspecialchars($selectedUser['email']); ?>
                        </p>
                    </div>

                    <!-- Messages -->
                    <div class="chat-messages" id="chatMessages">
                        <?php if (count($messages) > 0): ?>
                            <?php foreach ($messages as $msg): ?>
                                <div class="chat-message <?php echo $msg['sender_type']; ?>" data-message-id="<?php echo $msg['id']; ?>">
                                    <div><?php echo htmlspecialchars($msg['message']); ?></div>
                                    <div class="chat-message-time">
                                        <?php echo date('h:i A', strtotime($msg['created_at'])); ?>
                                        <?php 
                                        $isRead = $msg['is_read'] == 1;
                                        $readStatus = $isRead ? '✓ Seen' : '✓ Not Seen';
                                        $readClass = $isRead ? 'read-status seen' : 'read-status not-seen';
                                        ?>
                                        <span class="<?php echo $readClass; ?>" data-read-indicator="<?php echo $msg['id']; ?>"><?php echo $readStatus; ?></span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div style="text-align: center; opacity: 0.5; margin-top: 50px;">
                                No messages yet. Start the conversation!
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Input Area -->
                    <div class="chat-input-area">
                        <input type="text" id="messageInput" placeholder="Type your message..." 
                               onkeypress="if(event.key==='Enter') sendMessage()">
                        <button onclick="sendMessage()" class="chat-send-btn">Send</button>
                    </div>
                <?php else: ?>
                    <div class="no-chat-selected">
                        <div style="text-align: center;">
                            <div style="font-size: 64px; margin-bottom: 20px;">💬</div>
                            <h3>Select a chat to start messaging</h3>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <script src="../assets/js/toast.js"></script>
    <script>
        const selectedUserId = <?php echo $selectedUserId ?: 0; ?>;
        let lastMessageId = <?php echo count($messages) > 0 ? end($messages)['id'] : 0; ?>;

        function sendMessage() {
            const input = document.getElementById('messageInput');
            const message = input.value.trim();

            if (!message || !selectedUserId) return;

            const formData = new FormData();
            formData.append('message', message);
            formData.append('user_id', selectedUserId);

            fetch('../api/chat.php?action=send', {
                method: 'POST',
                body: formData
            })
            .then(res => res.json())
            .then(data => {
                if (data.success) {
                    input.value = '';
                    loadNewMessages();
                    showSuccess('Message sent!');
                } else {
                    showError('Failed to send message');
                }
            })
            .catch(err => {
                console.error('Send error:', err);
                showError('Error sending message');
            });
        }

        function loadNewMessages() {
            if (!selectedUserId) return;

            fetch(`../api/chat.php?action=poll&user_id=${selectedUserId}&last_id=${lastMessageId}`)
                .then(res => res.json())
                .then(data => {
                    if (data.messages && data.messages.length > 0) {
                        const messagesDiv = document.getElementById('chatMessages');
                        data.messages.forEach(msg => {
                            // Check if message already exists
                            const existingMsg = document.querySelector(`[data-message-id="${msg.id}"]`);
                            if (existingMsg) {
                                updateMessageReadStatus(msg.id, msg.is_read);
                                return;
                            }
                            
                            const messageDiv = document.createElement('div');
                            messageDiv.className = `chat-message ${msg.sender_type}`;
                            messageDiv.setAttribute('data-message-id', msg.id);
                            
                            const time = new Date(msg.created_at).toLocaleTimeString('en-US', {
                                hour: '2-digit',
                                minute: '2-digit'
                            });
                            
                            const isRead = msg.is_read == 1;
                            const readStatus = isRead ? '✓ Seen' : '✓ Not Seen';
                            const readClass = isRead ? 'read-status seen' : 'read-status not-seen';
                            
                            messageDiv.innerHTML = `
                                <div>${msg.message}</div>
                                <div class="chat-message-time">
                                    ${time}
                                    <span class="${readClass}" data-read-indicator="${msg.id}">${readStatus}</span>
                                </div>
                            `;
                            
                            messagesDiv.appendChild(messageDiv);
                            lastMessageId = Math.max(lastMessageId, msg.id);
                        });
                        
                        messagesDiv.scrollTop = messagesDiv.scrollHeight;
                    }
                });
        }
        
        function updateMessageReadStatus(messageId, isRead) {
            const indicator = document.querySelector(`[data-read-indicator="${messageId}"]`);
            if (indicator) {
                const wasRead = indicator.classList.contains('seen');
                const nowRead = isRead == 1;
                
                if (wasRead !== nowRead) {
                    indicator.className = nowRead ? 'read-status seen' : 'read-status not-seen';
                    indicator.textContent = nowRead ? '✓ Seen' : '✓ Not Seen';
                }
            }
        }
        
        function checkReadStatusUpdates() {
            if (!selectedUserId) return;
            
            fetch(`../api/chat.php?action=get&user_id=${selectedUserId}`)
                .then(res => res.json())
                .then(data => {
                    if (data.messages && data.messages.length > 0) {
                        data.messages.forEach(msg => {
                            updateMessageReadStatus(msg.id, msg.is_read);
                        });
                    }
                });
        }

        // Poll for new messages every 3 seconds
        if (selectedUserId) {
            setInterval(loadNewMessages, 3000);
            setInterval(checkReadStatusUpdates, 3000);
            
            // Scroll to bottom on load
            const messagesDiv = document.getElementById('chatMessages');
            if (messagesDiv) {
                messagesDiv.scrollTop = messagesDiv.scrollHeight;
            }
        }
    </script>
</body>
</html>
