<?php
session_start();
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../app/models/AIChatbot.php';

// Check if admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit;
}

$aiChatbot = new AIChatbot();
$settings = $aiChatbot->getSettings();
$stats = $aiChatbot->getStats(7);

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_settings'])) {
    $updateData = [
        'is_enabled' => isset($_POST['is_enabled']) ? 1 : 0,
        'auto_reply_delay' => intval($_POST['auto_reply_delay']),
        'ai_provider' => $_POST['ai_provider'],
        'api_key' => trim($_POST['api_key']),
        'system_prompt' => trim($_POST['system_prompt']),
        'max_tokens' => intval($_POST['max_tokens']),
        'temperature' => floatval($_POST['temperature'])
    ];
    
    if ($aiChatbot->updateSettings($updateData)) {
        $success = "Settings updated successfully!";
        $settings = $aiChatbot->getSettings();
    } else {
        $error = "Failed to update settings.";
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AI Chatbot Settings - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        .settings-container {
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .settings-grid {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 20px;
            margin-top: 20px;
        }
        
        .settings-card {
            background: var(--glass-bg);
            backdrop-filter: blur(20px);
            border: 1px solid var(--glass-border);
            border-radius: 20px;
            padding: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--text-light);
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 12px 15px;
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 10px;
            color: var(--text-light);
            font-size: 14px;
        }
        
        .form-group textarea {
            min-height: 120px;
            resize: vertical;
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: var(--neon-blue);
            background: rgba(255, 255, 255, 0.1);
        }
        
        .toggle-switch {
            position: relative;
            display: inline-block;
            width: 60px;
            height: 30px;
        }
        
        .toggle-switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }
        
        .toggle-slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(255, 255, 255, 0.1);
            transition: 0.4s;
            border-radius: 30px;
        }
        
        .toggle-slider:before {
            position: absolute;
            content: "";
            height: 22px;
            width: 22px;
            left: 4px;
            bottom: 4px;
            background-color: white;
            transition: 0.4s;
            border-radius: 50%;
        }
        
        input:checked + .toggle-slider {
            background: var(--primary-gradient);
        }
        
        input:checked + .toggle-slider:before {
            transform: translateX(30px);
        }
        
        .stat-card {
            background: rgba(255, 255, 255, 0.05);
            padding: 20px;
            border-radius: 15px;
            margin-bottom: 15px;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .stat-value {
            font-size: 32px;
            font-weight: 700;
            background: var(--primary-gradient);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin: 10px 0;
        }
        
        .stat-label {
            font-size: 14px;
            opacity: 0.7;
        }
        
        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .alert-success {
            background: rgba(74, 222, 128, 0.1);
            border: 1px solid rgba(74, 222, 128, 0.3);
            color: #4ade80;
        }
        
        .alert-error {
            background: rgba(245, 87, 108, 0.1);
            border: 1px solid rgba(245, 87, 108, 0.3);
            color: #f5576c;
        }
        
        .help-text {
            font-size: 12px;
            opacity: 0.6;
            margin-top: 5px;
        }
        
        @media (max-width: 768px) {
            .settings-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <div class="container">
            <a href="dashboard.php" class="navbar-brand">🔍 Admin Panel</a>
            <ul class="navbar-menu">
                <li><a href="dashboard.php">Dashboard</a></li>
                <li><a href="users.php">Users</a></li>
                <li><a href="plans.php">Plans</a></li>
                <li><a href="chat.php">Chat</a></li>
                <li><a href="chatbot-settings.php" class="active">🤖 AI Chatbot</a></li>
                <li><a href="../logout.php" class="btn btn-danger">Logout</a></li>
            </ul>
        </div>
    </nav>

    <div class="container mt-4 settings-container">
        <h2 class="gradient-text mb-4">🤖 AI Chatbot Settings</h2>

        <?php if (isset($success)): ?>
            <div class="alert alert-success">✅ <?php echo $success; ?></div>
        <?php endif; ?>
        
        <?php if (isset($error)): ?>
            <div class="alert alert-error">❌ <?php echo $error; ?></div>
        <?php endif; ?>

        <div class="settings-grid">
            <!-- Settings Form -->
            <div class="settings-card">
                <h3 class="gradient-text mb-3">Configuration</h3>
                
                <form method="POST" action="">
                    <div class="form-group">
                        <label>
                            <span style="display: flex; align-items: center; gap: 10px;">
                                Enable AI Chatbot
                                <label class="toggle-switch">
                                    <input type="checkbox" name="is_enabled" <?php echo $settings['is_enabled'] ? 'checked' : ''; ?>>
                                    <span class="toggle-slider"></span>
                                </label>
                            </span>
                        </label>
                        <div class="help-text">When enabled, bot will automatically respond to user messages</div>
                    </div>

                    <div class="form-group">
                        <label for="auto_reply_delay">Auto-Reply Delay (seconds)</label>
                        <input type="number" id="auto_reply_delay" name="auto_reply_delay" 
                               value="<?php echo $settings['auto_reply_delay']; ?>" min="0" max="300">
                        <div class="help-text">Bot will reply if admin hasn't responded within this time</div>
                    </div>

                    <div class="form-group">
                        <label for="ai_provider">AI Provider</label>
                        <select id="ai_provider" name="ai_provider">
                            <option value="gemini" <?php echo $settings['ai_provider'] === 'gemini' ? 'selected' : ''; ?>>Google Gemini</option>
                            <option value="openai" <?php echo $settings['ai_provider'] === 'openai' ? 'selected' : ''; ?>>OpenAI GPT</option>
                            <option value="custom" <?php echo $settings['ai_provider'] === 'custom' ? 'selected' : ''; ?>>Custom Rules</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="api_key">API Key</label>
                        <input type="password" id="api_key" name="api_key" 
                               value="<?php echo htmlspecialchars($settings['api_key']); ?>" 
                               placeholder="Enter your Gemini/OpenAI API key">
                        <div class="help-text">Get free API key from <a href="https://makersuite.google.com/app/apikey" target="_blank" style="color: var(--neon-blue);">Google AI Studio</a></div>
                    </div>

                    <div class="form-group">
                        <label for="system_prompt">System Prompt</label>
                        <textarea id="system_prompt" name="system_prompt" 
                                  placeholder="Define how the AI should behave..."><?php echo htmlspecialchars($settings['system_prompt']); ?></textarea>
                        <div class="help-text">Instructions for the AI about how to respond</div>
                    </div>

                    <div class="form-group">
                        <label for="max_tokens">Max Response Length (tokens)</label>
                        <input type="number" id="max_tokens" name="max_tokens" 
                               value="<?php echo $settings['max_tokens']; ?>" min="50" max="500">
                        <div class="help-text">Shorter responses are faster and cheaper</div>
                    </div>

                    <div class="form-group">
                        <label for="temperature">Creativity (0-1)</label>
                        <input type="number" id="temperature" name="temperature" 
                               value="<?php echo $settings['temperature']; ?>" min="0" max="1" step="0.1">
                        <div class="help-text">0 = Focused, 1 = Creative</div>
                    </div>

                    <button type="submit" name="update_settings" class="btn btn-primary" style="width: 100%;">
                        💾 Save Settings
                    </button>
                </form>
            </div>

            <!-- Statistics -->
            <div>
                <div class="settings-card">
                    <h3 class="gradient-text mb-3">📊 Last 7 Days Stats</h3>
                    
                    <div class="stat-card">
                        <div class="stat-label">Total Conversations</div>
                        <div class="stat-value"><?php echo number_format($stats['total_conversations']); ?></div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-label">Quick Responses</div>
                        <div class="stat-value"><?php echo number_format($stats['quick_responses']); ?></div>
                        <div class="help-text">Instant keyword-based replies</div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-label">AI Responses</div>
                        <div class="stat-value"><?php echo number_format($stats['ai_responses']); ?></div>
                        <div class="help-text">Generated by AI model</div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-label">Avg Response Time</div>
                        <div class="stat-value"><?php echo number_format($stats['avg_response_time']); ?>ms</div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-label">Avg Confidence</div>
                        <div class="stat-value"><?php echo number_format($stats['avg_confidence'] * 100, 1); ?>%</div>
                    </div>
                </div>

                <div class="settings-card" style="margin-top: 20px;">
                    <h3 class="gradient-text mb-3">ℹ️ Quick Guide</h3>
                    <ul style="line-height: 2; opacity: 0.8;">
                        <li>✅ Enable chatbot to auto-respond</li>
                        <li>⏱️ Set delay before bot replies</li>
                        <li>🔑 Add your Gemini API key</li>
                        <li>📝 Customize system prompt</li>
                        <li>🎯 Adjust creativity level</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <script src="../assets/js/toast.js"></script>
</body>
</html>
