<?php
session_start();
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../app/models/Plan.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit;
}

$planModel = new Plan();
$message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        $planModel->createPlan($_POST['name'], $_POST['description'], $_POST['price'], $_POST['duration_days'], $_POST['search_limit']);
        $message = 'Plan created successfully';
    } elseif ($action === 'update') {
        $planModel->updatePlan($_POST['id'], $_POST['name'], $_POST['description'], $_POST['price'], $_POST['duration_days'], $_POST['search_limit'], $_POST['status']);
        $message = 'Plan updated successfully';
    } elseif ($action === 'delete') {
        $planModel->deletePlan($_POST['id']);
        $message = 'Plan deleted successfully';
    }
}

$plans = $planModel->getAllPlans();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Plans - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
</head>
<body>
    <nav class="navbar">
        <div class="container">
            <a href="dashboard.php" class="navbar-brand">🔐 Admin Panel</a>
            <ul class="navbar-menu">
                <li><a href="dashboard.php">Dashboard</a></li>
                <li><a href="users.php">Users</a></li>
                <li><a href="plans.php">Plans</a></li>
                <li><a href="subscriptions.php">Subscriptions</a></li>
                <li><a href="payments.php">Payments</a></li>
                <li><a href="searches.php">Searches</a></li>
                <li><a href="alerts.php">Alerts</a></li>
                <li><a href="../logout.php" class="btn btn-danger">Logout</a></li>
            </ul>
        </div>
    </nav>

    <div class="container mt-4">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px;">
            <h2 class="gradient-text">Manage Plans</h2>
            <button onclick="showCreatePlan()" class="btn btn-primary">+ Create New Plan</button>
        </div>

        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Price</th>
                        <th>Duration</th>
                        <th>Search Limit</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($plans as $plan): ?>
                        <tr>
                            <td><?php echo $plan['id']; ?></td>
                            <td><?php echo htmlspecialchars($plan['name']); ?></td>
                            <td>₹<?php echo number_format($plan['price'], 2); ?></td>
                            <td><?php echo $plan['duration_days']; ?> days</td>
                            <td><?php echo $plan['search_limit']; ?></td>
                            <td>
                                <span class="badge <?php echo $plan['status'] === 'active' ? 'badge-success' : 'badge-danger'; ?>">
                                    <?php echo strtoupper($plan['status']); ?>
                                </span>
                            </td>
                            <td>
                                <button onclick='editPlan(<?php echo json_encode($plan); ?>)' class="btn btn-outline" style="padding: 5px 15px; font-size: 14px;">Edit</button>
                                <button onclick="deletePlan(<?php echo $plan['id']; ?>)" class="btn btn-danger" style="padding: 5px 15px; font-size: 14px;">Delete</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <script>
        function showCreatePlan() {
            Swal.fire({
                title: 'Create New Plan',
                html: `
                    <input id="name" class="form-control" placeholder="Plan Name" style="margin: 10px 0;">
                    <textarea id="description" class="form-control" placeholder="Description" style="margin: 10px 0;"></textarea>
                    <input id="price" type="number" class="form-control" placeholder="Price" style="margin: 10px 0;">
                    <input id="duration" type="number" class="form-control" placeholder="Duration (days)" style="margin: 10px 0;">
                    <input id="limit" type="number" class="form-control" placeholder="Search Limit" style="margin: 10px 0;">
                `,
                showCancelButton: true,
                confirmButtonText: 'Create',
                background: 'rgba(10, 14, 39, 0.95)',
                color: '#fff',
                confirmButtonColor: '#667eea',
                preConfirm: () => {
                    return {
                        name: document.getElementById('name').value,
                        description: document.getElementById('description').value,
                        price: document.getElementById('price').value,
                        duration_days: document.getElementById('duration').value,
                        search_limit: document.getElementById('limit').value
                    };
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <input type="hidden" name="action" value="create">
                        <input type="hidden" name="name" value="${result.value.name}">
                        <input type="hidden" name="description" value="${result.value.description}">
                        <input type="hidden" name="price" value="${result.value.price}">
                        <input type="hidden" name="duration_days" value="${result.value.duration_days}">
                        <input type="hidden" name="search_limit" value="${result.value.search_limit}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }

        function editPlan(plan) {
            Swal.fire({
                title: 'Edit Plan',
                html: `
                    <input id="name" class="form-control" value="${plan.name}" style="margin: 10px 0;">
                    <textarea id="description" class="form-control" style="margin: 10px 0;">${plan.description}</textarea>
                    <input id="price" type="number" class="form-control" value="${plan.price}" style="margin: 10px 0;">
                    <input id="duration" type="number" class="form-control" value="${plan.duration_days}" style="margin: 10px 0;">
                    <input id="limit" type="number" class="form-control" value="${plan.search_limit}" style="margin: 10px 0;">
                    <select id="status" class="form-control" style="margin: 10px 0;">
                        <option value="active" ${plan.status === 'active' ? 'selected' : ''}>Active</option>
                        <option value="inactive" ${plan.status === 'inactive' ? 'selected' : ''}>Inactive</option>
                    </select>
                `,
                showCancelButton: true,
                confirmButtonText: 'Update',
                background: 'rgba(10, 14, 39, 0.95)',
                color: '#fff',
                confirmButtonColor: '#667eea',
                preConfirm: () => {
                    return {
                        id: plan.id,
                        name: document.getElementById('name').value,
                        description: document.getElementById('description').value,
                        price: document.getElementById('price').value,
                        duration_days: document.getElementById('duration').value,
                        search_limit: document.getElementById('limit').value,
                        status: document.getElementById('status').value
                    };
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <input type="hidden" name="action" value="update">
                        <input type="hidden" name="id" value="${result.value.id}">
                        <input type="hidden" name="name" value="${result.value.name}">
                        <input type="hidden" name="description" value="${result.value.description}">
                        <input type="hidden" name="price" value="${result.value.price}">
                        <input type="hidden" name="duration_days" value="${result.value.duration_days}">
                        <input type="hidden" name="search_limit" value="${result.value.search_limit}">
                        <input type="hidden" name="status" value="${result.value.status}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }

        function deletePlan(id) {
            Swal.fire({
                title: 'Are you sure?',
                text: 'This will delete the plan permanently!',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonText: 'Yes, delete it!',
                background: 'rgba(10, 14, 39, 0.95)',
                color: '#fff',
                confirmButtonColor: '#f5576c'
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <input type="hidden" name="action" value="delete">
                        <input type="hidden" name="id" value="${id}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }

        <?php if ($message): ?>
        Swal.fire({
            icon: 'success',
            title: 'Success!',
            text: '<?php echo $message; ?>',
            background: 'rgba(10, 14, 39, 0.95)',
            color: '#fff',
            confirmButtonColor: '#667eea'
        });
        <?php endif; ?>
    </script>
</body>
</html>
