<?php
session_start();
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../app/models/Database.php';

// Check if admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit;
}

$db = Database::getInstance()->getConnection();

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $allowRepeatViews = isset($_POST['allow_repeat_views']) ? '1' : '0';
    
    $stmt = $db->prepare("INSERT INTO search_settings (setting_key, setting_value) VALUES ('allow_repeat_views', ?) ON DUPLICATE KEY UPDATE setting_value = ?");
    $stmt->execute([$allowRepeatViews, $allowRepeatViews]);
    
    $success = 'Settings updated successfully!';
}

// Get current settings
$stmt = $db->query("SELECT setting_value FROM search_settings WHERE setting_key = 'allow_repeat_views'");
$allowRepeatViews = $stmt->fetch()['setting_value'] ?? '1';

// Get all searches for locking
$searches = $db->query("
    SELECT s.*, u.name as user_name, u.email 
    FROM searches s 
    JOIN users u ON s.user_id = u.id 
    ORDER BY s.created_at DESC 
    LIMIT 50
")->fetchAll();

// Handle individual search lock/unlock
if (isset($_POST['toggle_lock'])) {
    $searchId = $_POST['search_id'];
    $currentLock = $_POST['current_lock'];
    $newLock = $currentLock == '1' ? '0' : '1';
    
    $stmt = $db->prepare("UPDATE searches SET admin_locked = ? WHERE id = ?");
    $stmt->execute([$newLock, $searchId]);
    
    header('Location: settings.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Search Settings - Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
</head>
<body>
    <nav class="navbar">
        <div class="container">
            <a href="dashboard.php" class="navbar-brand">🔍 Admin Panel</a>
            <ul class="navbar-menu">
                <li><a href="dashboard.php">Dashboard</a></li>
                <li><a href="users.php">Users</a></li>
                <li><a href="plans.php">Plans</a></li>
                <li><a href="payments.php">Payments</a></li>
                <li><a href="searches.php">Searches</a></li>
                <li><a href="settings.php">Settings</a></li>
                <li><a href="../logout.php" class="btn btn-danger">Logout</a></li>
            </ul>
        </div>
    </nav>

    <div class="container mt-4">
        <h2 class="gradient-text mb-4">⚙️ Search View Settings</h2>

        <?php if (isset($success)): ?>
            <div class="alert alert-success mb-4"><?php echo $success; ?></div>
        <?php endif; ?>

        <!-- Global Settings -->
        <div class="glass-card mb-4">
            <h3 class="gradient-text mb-3">🌐 Global Settings</h3>
            <form method="POST">
                <div class="mb-3">
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox" name="allow_repeat_views" value="1" <?php echo $allowRepeatViews == '1' ? 'checked' : ''; ?> style="width: 20px; height: 20px;">
                        <span>
                            <strong>Allow Repeat Views</strong><br>
                            <small style="opacity: 0.7;">If disabled, users can only view each search result once</small>
                        </span>
                    </label>
                </div>
                <button type="submit" class="btn btn-primary">Save Settings</button>
            </form>
        </div>

        <!-- Individual Search Locks -->
        <div class="glass-card">
            <h3 class="gradient-text mb-3">🔒 Individual Search Locks</h3>
            <p style="opacity: 0.8; margin-bottom: 20px;">Lock specific searches to prevent users from viewing them</p>
            
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>User</th>
                            <th>Type</th>
                            <th>Query</th>
                            <th>Views</th>
                            <th>Date</th>
                            <th>Status</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($searches as $search): ?>
                            <tr>
                                <td><?php echo $search['id']; ?></td>
                                <td>
                                    <strong><?php echo htmlspecialchars($search['user_name']); ?></strong><br>
                                    <small style="opacity: 0.7;"><?php echo htmlspecialchars($search['email']); ?></small>
                                </td>
                                <td>
                                    <span class="badge badge-info">
                                        <?php echo strtoupper($search['search_type']); ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($search['query_value']); ?></td>
                                <td><?php echo $search['view_count']; ?></td>
                                <td><?php echo date('d M Y', strtotime($search['created_at'])); ?></td>
                                <td>
                                    <?php if ($search['admin_locked']): ?>
                                        <span class="badge badge-danger">🔒 Locked</span>
                                    <?php else: ?>
                                        <span class="badge badge-success">✅ Open</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="search_id" value="<?php echo $search['id']; ?>">
                                        <input type="hidden" name="current_lock" value="<?php echo $search['admin_locked']; ?>">
                                        <button type="submit" name="toggle_lock" class="btn <?php echo $search['admin_locked'] ? 'btn-success' : 'btn-danger'; ?>" style="padding: 5px 15px; font-size: 14px;">
                                            <?php echo $search['admin_locked'] ? '🔓 Unlock' : '🔒 Lock'; ?>
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <script src="../assets/js/toast.js"></script>
    <script>
        <?php if (isset($success)): ?>
        showSuccess('<?php echo addslashes($success); ?>');
        <?php endif; ?>
    </script>
</body>
</html>
