<?php
session_start();
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../app/models/User.php';
require_once __DIR__ . '/../../app/models/Subscription.php';

// Check if user is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit;
}

$userModel = new User();
$subscriptionModel = new Subscription();

// Handle actions
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $userId = $_POST['user_id'] ?? 0;

    switch ($action) {
        case 'block':
            if ($userModel->updateUserStatus($userId, 'blocked')) {
                $message = 'User blocked successfully';
                $messageType = 'success';
            }
            break;
        case 'unblock':
            if ($userModel->updateUserStatus($userId, 'active')) {
                $message = 'User unblocked successfully';
                $messageType = 'success';
            }
            break;
        case 'make_admin':
            if ($userModel->updateUserRole($userId, 'admin')) {
                $message = 'User promoted to admin';
                $messageType = 'success';
            }
            break;
        case 'make_user':
            if ($userModel->updateUserRole($userId, 'user')) {
                $message = 'Admin demoted to user';
                $messageType = 'success';
            }
            break;
        case 'assign_plan':
            $planId = $_POST['plan_id'] ?? 0;
            if ($subscriptionModel->assignPlanToUser($userId, $planId)) {
                $message = 'Plan assigned successfully';
                $messageType = 'success';
            }
            break;
    }
}

$users = $userModel->getAllUsers();

// Get plans for assignment
require_once __DIR__ . '/../../app/models/Plan.php';
$planModel = new Plan();
$plans = $planModel->getActivePlans();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Users - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="container">
            <a href="dashboard.php" class="navbar-brand">🔐 Admin Panel</a>
            <ul class="navbar-menu">
                <li><a href="dashboard.php">Dashboard</a></li>
                <li><a href="users.php">Users</a></li>
                <li><a href="plans.php">Plans</a></li>
                <li><a href="subscriptions.php">Subscriptions</a></li>
                <li><a href="payments.php">Payments</a></li>
                <li><a href="searches.php">Searches</a></li>
                <li><a href="alerts.php">Alerts</a></li>
                <li><a href="../logout.php" class="btn btn-danger">Logout</a></li>
            </ul>
        </div>
    </nav>

    <div class="container mt-4">
        <h2 class="gradient-text mb-4">Manage Users</h2>

        <?php if ($message): ?>
            <div class="alert alert-<?php echo $messageType === 'success' ? 'success' : 'danger'; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <div class="table-responsive">
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Email</th>
                        <th>Role</th>
                        <th>Status</th>
                        <th>Joined</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $user): ?>
                        <tr>
                            <td><?php echo $user['id']; ?></td>
                            <td><?php echo htmlspecialchars($user['name']); ?></td>
                            <td><?php echo htmlspecialchars($user['email']); ?></td>
                            <td>
                                <span class="badge <?php echo $user['role'] === 'admin' ? 'badge-danger' : 'badge-info'; ?>">
                                    <?php echo strtoupper($user['role']); ?>
                                </span>
                            </td>
                            <td>
                                <span class="badge <?php echo $user['status'] === 'active' ? 'badge-success' : 'badge-danger'; ?>">
                                    <?php echo strtoupper($user['status']); ?>
                                </span>
                            </td>
                            <td><?php echo date('d M Y', strtotime($user['created_at'])); ?></td>
                            <td>
                                <?php if ($user['id'] !== $_SESSION['user_id']): ?>
                                    <button onclick="showActions(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['name']); ?>', '<?php echo $user['status']; ?>', '<?php echo $user['role']; ?>')" 
                                            class="btn btn-outline" style="padding: 5px 15px; font-size: 14px;">
                                        Actions
                                    </button>
                                <?php else: ?>
                                    <span style="opacity: 0.5;">Current User</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <script>
        const plans = <?php echo json_encode($plans); ?>;

        function showActions(userId, userName, status, role) {
            let actions = {};

            // Status actions
            if (status === 'active') {
                actions['Block User'] = () => performAction('block', userId);
            } else {
                actions['Unblock User'] = () => performAction('unblock', userId);
            }

            // Role actions
            if (role === 'user') {
                actions['Make Admin'] = () => performAction('make_admin', userId);
            } else {
                actions['Make User'] = () => performAction('make_user', userId);
            }

            // Assign plan
            actions['Assign Plan'] = () => showPlanSelection(userId);

            Swal.fire({
                title: 'Actions for ' + userName,
                showCancelButton: true,
                showConfirmButton: false,
                cancelButtonText: 'Close',
                html: Object.keys(actions).map(action => 
                    `<button class="btn btn-primary" style="margin: 5px; width: 200px;" onclick="window.currentActions['${action}']()">${action}</button>`
                ).join('<br>'),
                background: 'rgba(10, 14, 39, 0.95)',
                color: '#fff'
            });

            window.currentActions = actions;
        }

        function performAction(action, userId) {
            Swal.close();
            
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="${action}">
                <input type="hidden" name="user_id" value="${userId}">
            `;
            document.body.appendChild(form);
            form.submit();
        }

        function showPlanSelection(userId) {
            Swal.close();

            let planOptions = plans.map(plan => 
                `<option value="${plan.id}">${plan.name} - ₹${plan.price} (${plan.search_limit} searches)</option>`
            ).join('');

            Swal.fire({
                title: 'Assign Plan',
                html: `
                    <select id="planSelect" class="form-control">
                        <option value="">Select a plan</option>
                        ${planOptions}
                    </select>
                `,
                showCancelButton: true,
                confirmButtonText: 'Assign',
                background: 'rgba(10, 14, 39, 0.95)',
                color: '#fff',
                confirmButtonColor: '#667eea',
                preConfirm: () => {
                    const planId = document.getElementById('planSelect').value;
                    if (!planId) {
                        Swal.showValidationMessage('Please select a plan');
                        return false;
                    }
                    return planId;
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.innerHTML = `
                        <input type="hidden" name="action" value="assign_plan">
                        <input type="hidden" name="user_id" value="${userId}">
                        <input type="hidden" name="plan_id" value="${result.value}">
                    `;
                    document.body.appendChild(form);
                    form.submit();
                }
            });
        }

        <?php if ($message): ?>
        Swal.fire({
            icon: '<?php echo $messageType === 'success' ? 'success' : 'error'; ?>',
            title: '<?php echo $messageType === 'success' ? 'Success!' : 'Error'; ?>',
            text: '<?php echo $message; ?>',
            background: 'rgba(10, 14, 39, 0.95)',
            color: '#fff',
            confirmButtonColor: '#667eea'
        });
        <?php endif; ?>
    </script>
</body>
</html>
