<?php
// Start output buffering to prevent any stray output
ob_start();

// Disable display_errors to prevent warnings from breaking JSON
ini_set('display_errors', 0);
error_reporting(E_ALL);

session_start();
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../app/models/Chat.php';
require_once __DIR__ . '/../../app/models/AIChatbot.php';

// Clear any output that might have been generated
ob_clean();

// Set JSON header first
header('Content-Type: application/json');

// Enable error logging
error_log("Chat API called - User ID: " . ($_SESSION['user_id'] ?? 'none') . ", Action: " . ($_GET['action'] ?? 'none'));

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    error_log("Chat API: User not logged in");
    echo json_encode(['error' => 'Not logged in', 'success' => false]);
    exit;
}

$chatModel = new Chat();
$action = $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'send':
            $message = trim($_POST['message'] ?? '');
            if (empty($message)) {
                error_log("Chat API: Empty message");
                echo json_encode(['error' => 'Message is required', 'success' => false]);
                exit;
            }
            
            $senderType = $_SESSION['role'] === 'admin' ? 'admin' : 'user';
            $userId = $_SESSION['role'] === 'admin' ? ($_POST['user_id'] ?? 0) : $_SESSION['user_id'];
            
            error_log("Chat API: Sending message - User: $userId, Type: $senderType, Message: $message");
            
            $result = $chatModel->sendMessage($userId, $senderType, $message);
            
            error_log("Chat API: Send result - " . ($result ? 'SUCCESS' : 'FAILED'));
            
            // Send success response immediately
            echo json_encode(['success' => $result, 'message' => $result ? 'Message sent' : 'Failed to send']);
            
            // AI Chatbot Auto-Response (only for user messages) - runs after response sent
            if ($result && $senderType === 'user') {
                // Flush output to send response to user immediately
                if (function_exists('fastcgi_finish_request')) {
                    fastcgi_finish_request();
                }
                
                $aiChatbot = new AIChatbot();
                if ($aiChatbot->shouldAutoReply($userId)) {
                    // Generate AI response
                    $botResponse = $aiChatbot->generateResponse($userId, $message);
                    $aiChatbot->sendBotMessage($userId, $botResponse);
                    
                    error_log("Chat API: AI Bot responded to user $userId");
                }
            }
            
            break;
            
        case 'get':
            $userId = $_SESSION['role'] === 'admin' ? ($_GET['user_id'] ?? 0) : $_SESSION['user_id'];
            $messages = $chatModel->getUserMessages($userId);
            
            error_log("Chat API: Get messages - User: $userId, Count: " . count($messages));
            
            // Mark admin messages as read for user
            if ($_SESSION['role'] === 'user') {
                $chatModel->markAsRead($userId, 'admin');
            }
            
            echo json_encode(['messages' => $messages, 'success' => true]);
            break;
            
        case 'poll':
            $userId = $_SESSION['role'] === 'admin' ? ($_GET['user_id'] ?? 0) : $_SESSION['user_id'];
            $lastId = $_GET['last_id'] ?? 0;
            $newMessages = $chatModel->getNewMessages($userId, $lastId);
            
            if (count($newMessages) > 0) {
                error_log("Chat API: Poll - User: $userId, New messages: " . count($newMessages));
            }
            
            echo json_encode(['messages' => $newMessages, 'success' => true]);
            break;
            
        case 'unread':
            $count = $chatModel->getUnreadCount($_SESSION['user_id'], 'admin');
            echo json_encode(['count' => $count, 'success' => true]);
            break;
            
        default:
            error_log("Chat API: Invalid action - $action");
            echo json_encode(['error' => 'Invalid action', 'success' => false]);
    }
} catch (Exception $e) {
    error_log("Chat API Exception: " . $e->getMessage());
    echo json_encode(['error' => $e->getMessage(), 'success' => false]);
}
