<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/models/User.php';
require_once __DIR__ . '/../app/models/Subscription.php';
require_once __DIR__ . '/../app/models/Search.php';
require_once __DIR__ . '/../app/models/Payment.php';

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'user') {
    header('Location: login.php');
    exit;
}

$userModel = new User();
$subscriptionModel = new Subscription();
$searchModel = new Search();
$paymentModel = new Payment();

$user = $userModel->getUserById($_SESSION['user_id']);
$activeSubscription = $subscriptionModel->getActiveSubscription($_SESSION['user_id']);
$recentSearches = $searchModel->getUserSearches($_SESSION['user_id'], 10);
$recentPayments = $paymentModel->getUserPayments($_SESSION['user_id']);
$remainingSearches = $subscriptionModel->getRemainingSearches($_SESSION['user_id']);

// Debug
error_log("Dashboard searches count: " . count($recentSearches));
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - <?php echo SITE_NAME; ?></title>
    
    <!-- Preconnect to external domains for faster loading -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link rel="preconnect" href="https://unpkg.com">
    <link rel="dns-prefetch" href="https://cdn.jsdelivr.net">
    
    <link rel="stylesheet" href="assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/user-navbar.php'; ?>

    <div class="container mt-4">
        <!-- Welcome Section -->
        <div class="glass-card mb-4" data-aos="fade-up">
            <h2 class="gradient-text">Welcome back, <?php echo htmlspecialchars($user['name']); ?>! 👋</h2>
            <p style="opacity: 0.8;">Here's your account overview</p>
        </div>

        <!-- Stats Cards -->
        <div class="row">
            <?php if ($activeSubscription): ?>
                <div class="col-md-4" data-aos="fade-up" data-aos-delay="100">
                    <div class="stat-card">
                        <h3><?php echo htmlspecialchars($activeSubscription['plan_name']); ?></h3>
                        <p>Active Plan</p>
                        <span class="badge badge-success">Active</span>
                    </div>
                </div>
                <div class="col-md-4" data-aos="fade-up" data-aos-delay="200">
                    <div class="stat-card">
                        <h3 class="neon-text"><?php echo $remainingSearches; ?></h3>
                        <p>Searches Remaining</p>
                        <small style="opacity: 0.7;">of <?php echo $activeSubscription['total_search_limit']; ?> total</small>
                    </div>
                </div>
                <div class="col-md-4" data-aos="fade-up" data-aos-delay="300">
                    <div class="stat-card">
                        <h3><?php echo date('d M Y', strtotime($activeSubscription['end_date'])); ?></h3>
                        <p>Valid Until</p>
                    </div>
                </div>
            <?php else: ?>
                <div class="col-md-12" data-aos="fade-up">
                    <div class="alert alert-danger">
                        <h3>No Active Subscription</h3>
                        <p>You don't have an active subscription. Please purchase a plan to start searching.</p>
                        <a href="plans.php" class="btn btn-primary mt-2">View Plans</a>
                    </div>
                </div>
            <?php endif; ?>
        </div>

        <!-- Quick Actions -->
        <div class="row mt-4">
            <div class="col-md-6" data-aos="fade-right">
                <div class="glass-card text-center">
                    <div style="font-size: 64px; margin-bottom: 20px;">📱</div>
                    <h3>Mobile Lookup</h3>
                    <p>Search for mobile number information</p>
                    <a href="search.php?type=mobile" class="btn btn-primary mt-2">Search Now</a>
                </div>
            </div>
            <div class="col-md-6" data-aos="fade-left">
                <div class="glass-card text-center">
                    <div style="font-size: 64px; margin-bottom: 20px;">🆔</div>
                    <h3>Aadhaar Lookup</h3>
                    <p>Search for Aadhaar information</p>
                    <a href="search.php?type=aadhaar" class="btn btn-primary mt-2">Search Now</a>
                </div>
            </div>
        </div>

        <!-- Recent Searches -->
        <div class="mt-4" data-aos="fade-up">
            <h3 class="gradient-text mb-3">Recent Searches (<?php echo count($recentSearches); ?>)</h3>
            <?php if (count($recentSearches) > 0): ?>
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th>Type</th>
                                <th>Query</th>
                                <th>Source</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recentSearches as $search): ?>
                                <tr>
                                    <td>
                                        <span class="badge badge-info">
                                            <?php echo strtoupper($search['search_type']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($search['query_value']); ?></td>
                                    <td>
                                        <span class="badge <?php echo $search['source'] === 'cache' ? 'badge-success' : 'badge-warning'; ?>">
                                            <?php echo strtoupper($search['source']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('d M Y H:i', strtotime($search['created_at'])); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <div class="text-center mt-3">
                    <a href="history.php" class="btn btn-outline">View All History</a>
                </div>
            <?php else: ?>
                <div class="glass-card text-center">
                    <p style="opacity: 0.7;">No searches yet. Start your first search now!</p>
                </div>
            <?php endif; ?>
        </div>

        <!-- Recent Transactions -->
        <div class="mt-4" data-aos="fade-up">
            <h3 class="gradient-text mb-3">💳 Recent Transactions</h3>
            <?php if (count($recentPayments) > 0): ?>
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th>Plan</th>
                                <th>Amount</th>
                                <th>Payment ID</th>
                                <th>Status</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach (array_slice($recentPayments, 0, 5) as $payment): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo htmlspecialchars($payment['plan_name']); ?></strong>
                                    </td>
                                    <td>
                                        <span style="color: var(--neon-blue); font-weight: 600;">
                                            ₹<?php echo number_format($payment['amount'], 2); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <small style="opacity: 0.8;">
                                            <?php echo $payment['razorpay_payment_id'] ? substr($payment['razorpay_payment_id'], 0, 20) . '...' : 'N/A'; ?>
                                        </small>
                                    </td>
                                    <td>
                                        <span class="badge <?php 
                                            echo $payment['status'] === 'success' ? 'badge-success' : 
                                                ($payment['status'] === 'pending' ? 'badge-warning' : 'badge-danger'); 
                                        ?>">
                                            <?php echo ucfirst($payment['status']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('d M Y H:i', strtotime($payment['created_at'])); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="glass-card text-center">
                    <p style="opacity: 0.7;">No transactions yet. Purchase a plan to get started!</p>
                    <a href="plans.php" class="btn btn-primary mt-2">View Plans</a>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <!-- Chat Widget -->
    <?php include 'includes/chat-widget.php'; ?>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="assets/js/toast.js"></script>
    <script>
        AOS.init({
            duration: 1000,
            once: true
        });
        
        // Check for login success from session
        <?php if (isset($_SESSION['login_success']) && $_SESSION['login_success']): ?>
            showSuccess('🎉 Welcome back, <?php echo htmlspecialchars($_SESSION['login_user_name'] ?? 'User'); ?>! Login successful!', 5000);
            <?php 
            unset($_SESSION['login_success']);
            unset($_SESSION['login_user_name']);
            ?>
        <?php endif; ?>
        
        // Check for payment success from session
        <?php if (isset($_SESSION['payment_success']) && $_SESSION['payment_success']): ?>
            showSuccess('🎉 Payment Successful! Your <?php echo htmlspecialchars($_SESSION['payment_plan'] ?? 'plan'); ?> subscription (₹<?php echo number_format($_SESSION['payment_amount'] ?? 0, 2); ?>) has been activated!', 6000);
            <?php 
            unset($_SESSION['payment_success']);
            unset($_SESSION['payment_plan']);
            unset($_SESSION['payment_amount']);
            ?>
        <?php endif; ?>
    </script>
</body>
</html>
