<!-- Chat Widget -->
<div id="chatWidget" class="chat-widget">
    <div class="chat-toggle" onclick="toggleChat()">
        <span class="chat-icon">💬</span>
        <span class="chat-badge" id="chatBadge" style="display: none;">0</span>
    </div>
    
    <div class="chat-box" id="chatBox" style="display: none;">
        <div class="chat-header">
            <h4>💬 Chat with Admin</h4>
            <div class="chat-header-actions">
                <button onclick="refreshChat()" class="chat-refresh" title="Refresh messages">🔄</button>
                <button onclick="toggleChat()" class="chat-close">×</button>
            </div>
        </div>
        
        <div class="chat-messages" id="chatMessages">
            <div class="chat-loading">Loading messages...</div>
        </div>
        
        <!-- Quick Action Buttons -->
        <div class="chat-quick-actions" id="chatQuickActions">
            <div class="quick-actions-header" onclick="toggleQuickActions()">
                <span class="quick-actions-label">Quick Actions</span>
                <span class="quick-actions-toggle collapsed" id="quickActionsToggle">▶</span>
            </div>
            <div class="quick-actions-grid hidden" id="quickActionsGrid">
                <button class="quick-action-btn" onclick="sendQuickMessage('What is my plan status?')" title="View your subscription details">
                    <span class="quick-action-icon">📊</span>
                    <span class="quick-action-text">My Plan</span>
                </button>
                <button class="quick-action-btn" onclick="sendQuickMessage('Show my last 5 searches')" title="View recent search history">
                    <span class="quick-action-icon">🔍</span>
                    <span class="quick-action-text">Recent Searches</span>
                </button>
                <button class="quick-action-btn" onclick="sendQuickMessage('Show my profile details')" title="View your account information">
                    <span class="quick-action-icon">👤</span>
                    <span class="quick-action-text">My Profile</span>
                </button>
                <button class="quick-action-btn" onclick="sendQuickMessage('How many searches do I have left?')" title="Check remaining searches">
                    <span class="quick-action-icon">📈</span>
                    <span class="quick-action-text">Usage Stats</span>
                </button>
            </div>
        </div>
        
        <div class="chat-input">
            <input type="text" id="chatMessageInput" placeholder="Type your message..." onkeypress="if(event.key==='Enter') sendChatMessage()">
            <button onclick="sendChatMessage()" class="chat-send-btn">Send</button>
        </div>
    </div>
</div>

<style>
.chat-widget {
    position: fixed;
    bottom: 20px;
    right: 20px;
    z-index: 9999;
}

.chat-toggle {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    background: var(--primary-gradient);
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    box-shadow: 0 4px 20px rgba(102, 126, 234, 0.4);
    transition: all 0.3s ease;
    position: relative;
}

.chat-toggle:hover {
    transform: scale(1.1);
    box-shadow: 0 6px 30px rgba(102, 126, 234, 0.6);
}

.chat-icon {
    font-size: 28px;
}

.chat-badge {
    position: absolute;
    top: -5px;
    right: -5px;
    background: #f5576c;
    color: white;
    border-radius: 50%;
    width: 24px;
    height: 24px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 12px;
    font-weight: bold;
}

.chat-box {
    position: absolute;
    bottom: 80px;
    right: 0;
    width: 350px;
    height: 500px;
    background: var(--glass-bg);
    backdrop-filter: blur(20px);
    border: 1px solid var(--glass-border);
    border-radius: 20px;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
    display: flex;
    flex-direction: column;
    animation: slideUp 0.3s ease;
}

@keyframes slideUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.chat-header {
    padding: 20px;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.chat-header h4 {
    margin: 0;
    font-size: 18px;
    background: var(--primary-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
}

.chat-header-actions {
    display: flex;
    gap: 10px;
    align-items: center;
}

.chat-refresh {
    background: none;
    border: none;
    color: var(--text-light);
    font-size: 20px;
    cursor: pointer;
    padding: 0;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    transition: all 0.3s ease;
}

.chat-refresh:hover {
    background: rgba(255, 255, 255, 0.1);
    transform: rotate(180deg);
}

.chat-refresh:active {
    transform: rotate(360deg);
}

.chat-close {
    background: none;
    border: none;
    color: var(--text-light);
    font-size: 28px;
    cursor: pointer;
    padding: 0;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    transition: all 0.3s ease;
}

.chat-close:hover {
    background: rgba(255, 255, 255, 0.1);
}

.chat-messages {
    flex: 1;
    overflow-y: auto;
    padding: 20px;
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.chat-loading {
    text-align: center;
    opacity: 0.7;
    padding: 20px;
}

.chat-message {
    max-width: 80%;
    padding: 12px 16px;
    border-radius: 15px;
    word-wrap: break-word;
    animation: fadeIn 0.3s ease;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

.chat-message.user {
    align-self: flex-end;
    background: var(--primary-gradient);
    color: white;
    border-bottom-right-radius: 5px;
}

.chat-message.admin {
    align-self: flex-start;
    background: rgba(255, 255, 255, 0.1);
    border-bottom-left-radius: 5px;
}

.chat-message.bot {
    align-self: flex-start;
    background: linear-gradient(135deg, rgba(139, 92, 246, 0.2), rgba(59, 130, 246, 0.2));
    border: 1px solid rgba(139, 92, 246, 0.3);
    border-bottom-left-radius: 5px;
    position: relative;
}

.chat-message.bot::before {
    content: '🤖';
    position: absolute;
    top: -8px;
    left: -8px;
    font-size: 16px;
    background: rgba(139, 92, 246, 0.8);
    border-radius: 50%;
    width: 24px;
    height: 24px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.chat-message-time {
    font-size: 11px;
    opacity: 0.7;
    margin-top: 5px;
    display: flex;
    align-items: center;
    gap: 8px;
}

.read-status {
    font-size: 10px;
    padding: 2px 6px;
    border-radius: 8px;
    font-weight: 500;
}

.read-status.seen {
    color: #4ade80;
    background: rgba(74, 222, 128, 0.1);
}

.read-status.not-seen {
    color: #94a3b8;
    background: rgba(148, 163, 184, 0.1);
}

.chat-quick-actions {
    border-top: 1px solid rgba(255, 255, 255, 0.1);
    background: rgba(255, 255, 255, 0.02);
}

.quick-actions-header {
    padding: 12px 15px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    cursor: pointer;
    transition: all 0.3s ease;
    user-select: none;
}

.quick-actions-header:hover {
    background: rgba(255, 255, 255, 0.05);
}

.quick-actions-label {
    font-size: 12px;
    font-weight: 600;
    opacity: 0.8;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.quick-actions-toggle {
    font-size: 12px;
    opacity: 0.6;
    transition: transform 0.3s ease;
}

.quick-actions-toggle.collapsed {
    transform: rotate(-90deg);
}

.quick-actions-grid {
    padding: 0 15px 12px 15px;
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 8px;
    max-height: 200px;
    overflow: hidden;
    transition: max-height 0.3s ease, padding 0.3s ease;
}

.quick-actions-grid.hidden {
    max-height: 0;
    padding: 0 15px;
}

.quick-action-btn {
    background: linear-gradient(135deg, rgba(102, 126, 234, 0.1), rgba(118, 75, 162, 0.1));
    border: 1px solid rgba(102, 126, 234, 0.2);
    border-radius: 10px;
    padding: 10px 12px;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 8px;
    color: var(--text-light);
    font-size: 13px;
    font-weight: 500;
}

.quick-action-btn:hover {
    background: linear-gradient(135deg, rgba(102, 126, 234, 0.2), rgba(118, 75, 162, 0.2));
    border-color: rgba(102, 126, 234, 0.4);
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.2);
}

.quick-action-btn:active {
    transform: translateY(0);
}

.quick-action-icon {
    font-size: 18px;
    line-height: 1;
}

.quick-action-text {
    flex: 1;
    text-align: left;
    font-size: 12px;
}

.chat-input {
    padding: 15px;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
    display: flex;
    gap: 10px;
}

.chat-input input {
    flex: 1;
    padding: 12px 15px;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 25px;
    color: var(--text-light);
    font-size: 14px;
}

.chat-input input:focus {
    outline: none;
    border-color: var(--neon-blue);
    background: rgba(255, 255, 255, 0.1);
}

.chat-send-btn {
    padding: 12px 20px;
    background: var(--primary-gradient);
    border: none;
    border-radius: 25px;
    color: white;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.chat-send-btn:hover {
    transform: scale(1.05);
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
}

/* Mobile responsive */
@media (max-width: 768px) {
    .chat-box {
        width: calc(100vw - 40px);
        height: calc(100vh - 100px);
        right: -10px;
    }
    
    .quick-actions-grid {
        grid-template-columns: 1fr;
    }
    
    .quick-action-btn {
        padding: 12px;
    }
}
</style>

<script>
let chatOpen = false;
let lastMessageId = 0;
let chatPollInterval;

// Get API path based on current URL
const currentPath = window.location.pathname;
const basePath = currentPath.includes('/admin/') 
    ? '../api/' 
    : 'api/';

console.log('Chat Widget - Current Path:', currentPath);
console.log('Chat Widget - API Base:', basePath);

function toggleChat() {
    chatOpen = !chatOpen;
    const chatBox = document.getElementById('chatBox');
    chatBox.style.display = chatOpen ? 'flex' : 'none';
    
    if (chatOpen) {
        loadChatMessages();
        startChatPolling();
        document.getElementById('chatBadge').style.display = 'none';
    } else {
        stopChatPolling();
    }
}

function sendQuickMessage(message) {
    const input = document.getElementById('chatMessageInput');
    input.value = message;
    sendChatMessage();
}

function toggleQuickActions() {
    const grid = document.getElementById('quickActionsGrid');
    const toggle = document.getElementById('quickActionsToggle');
    
    if (grid.classList.contains('hidden')) {
        grid.classList.remove('hidden');
        toggle.classList.remove('collapsed');
        toggle.textContent = '▼';
    } else {
        grid.classList.add('hidden');
        toggle.classList.add('collapsed');
        toggle.textContent = '▶';
    }
}

function refreshChat() {
    console.log('Refreshing chat messages...');
    
    // Add spinning animation to refresh button
    const refreshBtn = document.querySelector('.chat-refresh');
    refreshBtn.style.transform = 'rotate(360deg)';
    
    // Reset rotation after animation
    setTimeout(() => {
        refreshBtn.style.transform = 'rotate(0deg)';
    }, 300);
    
    // Reload messages
    loadChatMessages();
}

function loadChatMessages() {
    fetch(basePath + 'chat.php?action=get')
        .then(res => res.json())
        .then(data => {
            console.log('Chat messages loaded:', data);
            const messagesDiv = document.getElementById('chatMessages');
            messagesDiv.innerHTML = '';
            
            if (data.messages && data.messages.length > 0) {
                data.messages.forEach(msg => {
                    appendMessage(msg);
                    lastMessageId = Math.max(lastMessageId, msg.id);
                });
                scrollToBottom();
            } else {
                messagesDiv.innerHTML = '<div class="chat-loading">No messages yet. Start a conversation!</div>';
            }
        })
        .catch(err => {
            console.error('Chat error:', err);
            document.getElementById('chatMessages').innerHTML = '<div class="chat-loading" style="color: #f5576c;">Error loading messages. Please refresh.</div>';
        });
}

function sendChatMessage() {
    const input = document.getElementById('chatMessageInput');
    const message = input.value.trim();
    
    console.log('=== SEND MESSAGE DEBUG ===');
    console.log('Message:', message);
    console.log('Base Path:', basePath);
    
    if (!message) {
        console.log('Empty message, returning');
        return;
    }
    
    const formData = new FormData();
    formData.append('message', message);
    
    const url = basePath + 'chat.php?action=send';
    console.log('Sending to URL:', url);
    
    fetch(url, {
        method: 'POST',
        body: formData
    })
    .then(res => {
        console.log('Response status:', res.status);
        console.log('Response OK:', res.ok);
        return res.json();
    })
    .then(data => {
        console.log('Send response:', data);
        if (data.success) {
            input.value = '';
            loadChatMessages();
            console.log('Message sent successfully!');
        } else {
            console.error('Failed to send:', data);
            alert('Failed to send message: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        console.error('Send error:', err);
        console.error('Error details:', err.message);
        alert('Error sending message. Check console for details.');
    });
}

function appendMessage(msg) {
    const messagesDiv = document.getElementById('chatMessages');
    
    // Check if message already exists
    const existingMsg = document.querySelector(`[data-message-id="${msg.id}"]`);
    if (existingMsg) {
        // Update read status only
        updateMessageReadStatus(msg.id, msg.is_read);
        return;
    }
    
    const messageDiv = document.createElement('div');
    // Add 'bot' class if message is from AI bot
    const messageClass = msg.is_bot == 1 ? 'bot' : msg.sender_type;
    messageDiv.className = `chat-message ${messageClass}`;
    messageDiv.setAttribute('data-message-id', msg.id);
    
    const time = new Date(msg.created_at).toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit'
    });
    
    // Determine read status
    const isRead = msg.is_read == 1;
    const readStatus = isRead ? '✓ Seen' : '✓ Not Seen';
    const readClass = isRead ? 'read-status seen' : 'read-status not-seen';
    
    messageDiv.innerHTML = `
        <div>${escapeHtml(msg.message)}</div>
        <div class="chat-message-time">
            ${time}
            <span class="${readClass}" data-read-indicator="${msg.id}">${readStatus}</span>
        </div>
    `;
    
    messagesDiv.appendChild(messageDiv);
}

function updateMessageReadStatus(messageId, isRead) {
    const indicator = document.querySelector(`[data-read-indicator="${messageId}"]`);
    if (indicator) {
        const wasRead = indicator.classList.contains('seen');
        const nowRead = isRead == 1;
        
        if (wasRead !== nowRead) {
            indicator.className = nowRead ? 'read-status seen' : 'read-status not-seen';
            indicator.textContent = nowRead ? '✓ Seen' : '✓ Not Seen';
        }
    }
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function scrollToBottom() {
    const messagesDiv = document.getElementById('chatMessages');
    messagesDiv.scrollTop = messagesDiv.scrollHeight;
}

function startChatPolling() {
    chatPollInterval = setInterval(() => {
        // Poll for new messages
        fetch(`${basePath}chat.php?action=poll&last_id=${lastMessageId}`)
            .then(res => res.json())
            .then(data => {
                if (data.messages && data.messages.length > 0) {
                    console.log('New messages:', data.messages.length);
                    data.messages.forEach(msg => {
                        appendMessage(msg);
                        lastMessageId = Math.max(lastMessageId, msg.id);
                    });
                    scrollToBottom();
                    
                    // Update badge if chat closed
                    if (!chatOpen) {
                        updateUnreadBadge();
                    }
                }
            })
            .catch(err => console.error('Poll error:', err));
        
        // Also check for read status updates of all messages
        if (chatOpen) {
            fetch(`${basePath}chat.php?action=get`)
                .then(res => res.json())
                .then(data => {
                    if (data.messages && data.messages.length > 0) {
                        data.messages.forEach(msg => {
                            updateMessageReadStatus(msg.id, msg.is_read);
                        });
                    }
                })
                .catch(err => console.error('Read status update error:', err));
        }
    }, 3000); // Poll every 3 seconds
}

function stopChatPolling() {
    if (chatPollInterval) {
        clearInterval(chatPollInterval);
    }
}

function updateUnreadBadge() {
    fetch(basePath + 'chat.php?action=unread')
        .then(res => res.json())
        .then(data => {
            const badge = document.getElementById('chatBadge');
            if (data.count > 0) {
                badge.textContent = data.count;
                badge.style.display = 'flex';
            } else {
                badge.style.display = 'none';
            }
        })
        .catch(err => console.error('Badge error:', err));
}

// Check for unread messages every 10 seconds
setInterval(updateUnreadBadge, 10000);
updateUnreadBadge();

// Auto-start polling when page loads
startChatPolling();
</script>
