<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/models/User.php';

// Redirect if already logged in
if (isset($_SESSION['user_id'])) {
    if ($_SESSION['role'] === 'admin') {
        header('Location: admin/dashboard.php');
    } else {
        header('Location: dashboard.php');
    }
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';

    if (empty($email) || empty($password)) {
        $error = 'All fields are required';
    } else {
        $userModel = new User();
        $result = $userModel->login($email, $password);
        
        if (is_array($result) && isset($result['error'])) {
            $error = $result['error'];
        } elseif ($result) {
            $_SESSION['user_id'] = $result['id'];
            $_SESSION['user_name'] = $result['name'];
            $_SESSION['user_email'] = $result['email'];
            $_SESSION['role'] = $result['role'];
            
            // Send login notification
            if ($result['role'] === 'user' && !empty($result['telegram_number'])) {
                try {
                    require_once __DIR__ . '/../app/controllers/Notification.php';
                    $notification = new Notification();
                    
                    $message = "🔐 <b>Login Successful!</b>\n\n";
                    $message .= "👤 Welcome back, {$result['name']}!\n\n";
                    $message .= "🕐 Time: " . date('d M Y, h:i A') . "\n";
                    $message .= "📱 IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'Unknown') . "\n\n";
                    $message .= "If this wasn't you, please secure your account immediately!";
                    
                    $notification->sendUserTelegram($result['telegram_number'], $message);
                } catch (Exception $e) {
                    error_log("Login notification error: " . $e->getMessage());
                }
            }
            
            // Log activity
            try {
                require_once __DIR__ . '/../app/models/ActivityLog.php';
                $activityLog = new ActivityLog();
                $activityLog->logActivity(
                    $result['id'],
                    'login',
                    'User logged in successfully',
                    ['method' => 'password', 'role' => $result['role']]
                );
            } catch (Exception $e) {
                error_log("Activity log error: " . $e->getMessage());
            }
            
            // Set login success flag for toast
            $_SESSION['login_success'] = true;
            $_SESSION['login_user_name'] = $result['name'];
            
            if ($result['role'] === 'admin') {
                header('Location: admin/dashboard.php');
            } else {
                header('Location: dashboard.php');
            }
            exit;
        } else {
            $error = 'Invalid email or password';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="container">
            <a href="index.php" class="navbar-brand">⚡ Gravity</a>
            <ul class="navbar-menu">
                <li><a href="index.php">Home</a></li>
                <li><a href="register.php" class="btn btn-primary">Register</a></li>
            </ul>
        </div>
    </nav>

    <!-- Login Form -->
    <div class="container" style="max-width: 500px; margin-top: 100px;">
        <div class="glass-card" data-aos="fade-up">
            <h2 class="text-center gradient-text mb-3">Welcome Back</h2>
            <p class="text-center mb-4" style="opacity: 0.8;">Login to access your dashboard</p>

            <?php if ($error): ?>
                <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>

            <form method="POST" action="">
                <div class="mb-2">
                    <label class="form-label">Email Address</label>
                    <input type="email" name="email" class="form-control" placeholder="Enter your email" required value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                </div>

                <div class="mb-3">
                    <label class="form-label">Password</label>
                    <input type="password" name="password" class="form-control" placeholder="Enter your password" required>
                </div>

                <button type="submit" class="btn btn-primary" style="width: 100%;">Login Now</button>
            </form>

            <p class="text-center mt-3" style="opacity: 0.8;">
                Don't have an account? <a href="register.php" style="color: var(--neon-blue);">Register here</a>
            </p>
        </div>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="assets/js/toast.js"></script>
    <script>
        AOS.init({
            duration: 1000,
            once: true
        });

        <?php if ($error): ?>
        showError('<?php echo addslashes($error); ?>');
        <?php endif; ?>
    </script>
</body>
</html>
