<?php
session_start();
require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../app/models/Payment.php';
require_once __DIR__ . '/../../app/models/Plan.php';
require_once __DIR__ . '/../../app/models/Subscription.php';
require_once __DIR__ . '/../../app/models/User.php';
require_once __DIR__ . '/../../app/controllers/Notification.php';
require_once __DIR__ . '/../../app/libraries/SimpleRazorpay.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$razorpayOrderId = $input['razorpay_order_id'] ?? '';
$razorpayPaymentId = $input['razorpay_payment_id'] ?? '';
$razorpaySignature = $input['razorpay_signature'] ?? '';
$planId = $input['plan_id'] ?? 0;

if (!$razorpayOrderId || !$razorpayPaymentId || !$razorpaySignature || !$planId) {
    echo json_encode(['success' => false, 'message' => 'Invalid parameters']);
    exit;
}

try {
    // Verify signature using SimpleRazorpay
    $razorpay = new SimpleRazorpay(RAZORPAY_KEY_ID, RAZORPAY_KEY_SECRET);
    
    $isValid = $razorpay->verifySignature($razorpayOrderId, $razorpayPaymentId, $razorpaySignature);
    
    if (!$isValid) {
        error_log("Payment verification failed for order: $razorpayOrderId");
        throw new Exception('Payment verification failed');
    }

    // Payment verified successfully
    $paymentModel = new Payment();
    
    // Get plan details
    $planModel = new Plan();
    $plan = $planModel->getPlanById($planId);
    
    if (!$plan) {
        error_log("Plan not found: $planId");
        throw new Exception('Invalid plan');
    }
    
    // Create payment record FIRST
    try {
        $paymentId = $paymentModel->createPayment(
            $_SESSION['user_id'],
            $planId,
            $razorpayOrderId,
            $plan['price']
        );
        error_log("Payment record created with ID: $paymentId for user: {$_SESSION['user_id']}");
    } catch (Exception $e) {
        error_log("Failed to create payment record: " . $e->getMessage());
        throw new Exception('Failed to create payment record');
    }
    
    // Update payment to success
    try {
        $updated = $paymentModel->updatePaymentSuccess($razorpayOrderId, $razorpayPaymentId, $razorpaySignature);
        if (!$updated) {
            error_log("Failed to update payment status for order: $razorpayOrderId");
        } else {
            error_log("Payment updated to success for order: $razorpayOrderId");
        }
    } catch (Exception $e) {
        error_log("Error updating payment: " . $e->getMessage());
    }

    // Create subscription
    $subscriptionModel = new Subscription();
    $startDate = date('Y-m-d H:i:s');
    $endDate = date('Y-m-d H:i:s', strtotime("+{$plan['duration_days']} days"));
    
    try {
        $subscriptionModel->createSubscription(
            $_SESSION['user_id'],
            $planId,
            $startDate,
            $endDate,
            $plan['search_limit']
        );
        error_log("Subscription created for user: {$_SESSION['user_id']}, plan: $planId");
    } catch (Exception $e) {
        error_log("Failed to create subscription: " . $e->getMessage());
        throw new Exception('Failed to create subscription');
    }

    // Get user and subscription for notifications
    $userModel = new User();
    $user = $userModel->getUserById($_SESSION['user_id']);
    $subscription = $subscriptionModel->getActiveSubscription($_SESSION['user_id']);
    $payment = $paymentModel->getPaymentByOrderId($razorpayOrderId);

    // Set success message for toast
    $_SESSION['payment_success'] = true;
    $_SESSION['payment_plan'] = $plan['name'];
    $_SESSION['payment_amount'] = $plan['price'];

    // Send response immediately (don't wait for notifications)
    echo json_encode([
        'success' => true,
        'message' => 'Payment verified and subscription activated',
        'payment_id' => $paymentId ?? null
    ]);
    
    // Flush output to send response to client immediately
    if (function_exists('fastcgi_finish_request')) {
        fastcgi_finish_request();
    }
    
    // Now send notifications in background (won't block user)
    try {
        $notification = new Notification();
        error_log("Sending notifications for user: {$user['name']}, Telegram: " . ($user['telegram_number'] ?? 'not set'));
        $notification->notifyPaymentSuccess($user, $plan, $payment);
        $notification->notifySubscriptionActivated($user, $plan, $subscription);
        error_log("Notifications sent successfully");
    } catch (Exception $e) {
        error_log("Notification error: " . $e->getMessage());
    }

} catch (Exception $e) {
    // Payment verification failed
    error_log("Payment verification error: " . $e->getMessage());
    
    if (isset($paymentModel) && isset($razorpayOrderId)) {
        try {
            $paymentModel->updatePaymentFailed($razorpayOrderId);
        } catch (Exception $ex) {
            error_log("Failed to update payment status to failed: " . $ex->getMessage());
        }
    }
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

