<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/models/Plan.php';
require_once __DIR__ . '/../app/models/Subscription.php';
require_once __DIR__ . '/../app/models/Payment.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$planModel = new Plan();
$subscriptionModel = new Subscription();
$paymentModel = new Payment();
$plans = $planModel->getActivePlans();
$activeSubscription = $subscriptionModel->getActiveSubscription($_SESSION['user_id']);
$hasActiveSubscription = $activeSubscription && is_array($activeSubscription);
$recentPayments = $paymentModel->getUserPayments($_SESSION['user_id']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Plans - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    <script src="https://checkout.razorpay.com/v1/checkout.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
</head>
<body>
    <?php include 'includes/user-navbar.php'; ?>

    <div class="container mt-4">
        <h2 class="text-center gradient-text mb-4" data-aos="fade-up">Choose Your Plan</h2>
        <p class="text-center mb-4" style="opacity: 0.8;" data-aos="fade-up" data-aos-delay="100">Select the perfect plan for your needs</p>

        <?php if ($hasActiveSubscription): ?>
            <div class="glass-card mb-4" data-aos="fade-up" style="background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%); border: 2px solid rgba(102, 126, 234, 0.3);">
                <div style="display: flex; align-items: center; gap: 20px;">
                    <div style="background: var(--primary-gradient); width: 80px; height: 80px; border-radius: 20px; display: flex; align-items: center; justify-content: center; flex-shrink: 0;">
                        <span style="font-size: 40px;">✨</span>
                    </div>
                    <div style="flex: 1;">
                        <h3 style="margin: 0 0 10px 0; background: var(--primary-gradient); -webkit-background-clip: text; -webkit-text-fill-color: transparent;">
                            Active Subscription
                        </h3>
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-top: 15px;">
                            <div>
                                <p style="margin: 0; opacity: 0.7; font-size: 12px; text-transform: uppercase; letter-spacing: 1px;">Current Plan</p>
                                <p style="margin: 5px 0 0 0; font-size: 18px; font-weight: 700; color: var(--neon-blue);">
                                    <?php echo htmlspecialchars($activeSubscription['plan_name']); ?>
                                </p>
                            </div>
                            <div>
                                <p style="margin: 0; opacity: 0.7; font-size: 12px; text-transform: uppercase; letter-spacing: 1px;">Valid Until</p>
                                <p style="margin: 5px 0 0 0; font-size: 18px; font-weight: 700;">
                                    <?php echo date('d M Y', strtotime($activeSubscription['end_date'])); ?>
                                </p>
                            </div>
                            <div>
                                <p style="margin: 0; opacity: 0.7; font-size: 12px; text-transform: uppercase; letter-spacing: 1px;">Searches Left</p>
                                <p style="margin: 5px 0 0 0; font-size: 18px; font-weight: 700; color: #00f2fe;">
                                    <?php echo $activeSubscription['total_search_limit'] - $activeSubscription['used_searches']; ?> / <?php echo $activeSubscription['total_search_limit']; ?>
                                </p>
                            </div>
                        </div>
                        <p style="margin: 15px 0 0 0; opacity: 0.8; font-size: 14px;">
                            💡 You can upgrade to a higher plan anytime!
                        </p>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <div class="row">
            <?php foreach ($plans as $index => $plan): ?>
                <div class="col-md-4" data-aos="fade-up" data-aos-delay="<?php echo ($index + 1) * 100; ?>">
                    <div class="plan-card">
                        <h3><?php echo htmlspecialchars($plan['name']); ?></h3>
                        <div class="price">₹<?php echo number_format($plan['price'], 0); ?></div>
                        <p><?php echo $plan['duration_days']; ?> Days</p>
                        
                        <ul>
                            <li><?php echo $plan['search_limit']; ?> Searches</li>
                            <li>Mobile Lookup</li>
                            <li>Aadhaar Lookup</li>
                            <li>Search History</li>
                            <li>Email Notifications</li>
                            <?php if ($plan['price'] >= 299): ?>
                                <li>Priority Support</li>
                            <?php endif; ?>
                            <?php if ($plan['price'] >= 999): ?>
                                <li>API Access</li>
                                <li>Advanced Analytics</li>
                            <?php endif; ?>
                        </ul>

                        <?php 
                        $isCurrentPlan = $hasActiveSubscription && $activeSubscription['plan_id'] == $plan['id'];
                        $buttonDisabled = $isCurrentPlan;
                        $buttonClass = $isCurrentPlan ? 'btn-outline' : 'btn-primary';
                        $buttonText = $isCurrentPlan ? '✓ Current Plan' : 'Buy Now';
                        $buttonStyle = $isCurrentPlan ? 'opacity: 0.7; cursor: not-allowed;' : '';
                        ?>
                        <button onclick="<?php echo $buttonDisabled ? 'return false;' : 'buyPlan(' . $plan['id'] . ', \'' . htmlspecialchars($plan['name']) . '\', ' . $plan['price'] . ')'; ?>" 
                                class="btn <?php echo $buttonClass; ?>" 
                                style="width: 100%; <?php echo $buttonStyle; ?>">
                            <?php echo $buttonText; ?>
                        </button>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Recent Transactions -->
        <?php if (count($recentPayments) > 0): ?>
            <div class="mt-5" data-aos="fade-up">
                <h3 class="gradient-text mb-4" style="text-align: center;">💳 Recent Transactions</h3>
                
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Plan</th>
                                <th>Amount</th>
                                <th>Payment ID</th>
                                <th>Status</th>
                                <th>Date & Time</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach (array_slice($recentPayments, 0, 10) as $index => $payment): ?>
                                <tr>
                                    <td><?php echo $index + 1; ?></td>
                                    <td>
                                        <strong style="color: var(--neon-blue);">
                                            <?php echo htmlspecialchars($payment['plan_name']); ?>
                                        </strong>
                                    </td>
                                    <td>
                                        <span style="color: #4ade80; font-weight: 600; font-size: 16px;">
                                            ₹<?php echo number_format($payment['amount'], 2); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <small style="opacity: 0.8; font-family: monospace; font-size: 12px;">
                                            <?php echo $payment['razorpay_payment_id'] ? substr($payment['razorpay_payment_id'], 0, 20) . '...' : 'N/A'; ?>
                                        </small>
                                    </td>
                                    <td>
                                        <span class="badge <?php 
                                            echo $payment['status'] === 'success' ? 'badge-success' : 
                                                ($payment['status'] === 'pending' ? 'badge-warning' : 'badge-danger'); 
                                        ?>">
                                            <?php 
                                            echo $payment['status'] === 'success' ? '✅ ' : 
                                                ($payment['status'] === 'pending' ? '⏳ ' : '❌ '); 
                                            echo ucfirst($payment['status']); 
                                            ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span style="font-size: 14px;">
                                            <?php echo date('d M Y', strtotime($payment['created_at'])); ?>
                                        </span>
                                        <br>
                                        <small style="opacity: 0.7; font-size: 12px;">
                                            <?php echo date('h:i A', strtotime($payment['created_at'])); ?>
                                        </small>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <?php if (count($recentPayments) > 10): ?>
                    <div class="text-center mt-3">
                        <p style="opacity: 0.7; font-size: 14px;">
                            Showing 10 of <?php echo count($recentPayments); ?> transactions
                        </p>
                    </div>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div class="mt-5" data-aos="fade-up">
                <div class="glass-card text-center" style="padding: 40px;">
                    <div style="font-size: 64px; margin-bottom: 20px; opacity: 0.5;">💳</div>
                    <h3 style="margin-bottom: 15px;">No Transactions Yet</h3>
                    <p style="opacity: 0.7; margin-bottom: 20px;">
                        You haven't made any payments yet. Purchase a plan to get started!
                    </p>
                    <p style="opacity: 0.8; font-size: 14px;">
                        💡 Your transaction history will appear here after your first purchase
                    </p>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="assets/js/toast.js"></script>
    <script>
        AOS.init({
            duration: 1000,
            once: true
        });

        function buyPlan(planId, planName, price) {
            // Create Razorpay order
            const formData = new FormData();
            formData.append('plan_id', planId);
            
            fetch('payment/create-order.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                console.log('Order response:', data);
                
                if (data.error) {
                    showError(data.error || data.message || 'Failed to create order');
                    return;
                }
                
                if (data.success && data.order_id) {
                    var options = {
                        "key": "<?php echo RAZORPAY_KEY_ID; ?>",
                        "amount": data.amount,
                        "currency": "INR",
                        "name": "<?php echo SITE_NAME; ?>",
                        "description": planName,
                        "order_id": data.order_id,
                        "handler": function (response) {
                            // Verify payment
                            fetch('payment/verify-payment.php', {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                },
                                body: JSON.stringify({
                                    razorpay_order_id: response.razorpay_order_id,
                                    razorpay_payment_id: response.razorpay_payment_id,
                                    razorpay_signature: response.razorpay_signature,
                                    plan_id: planId
                                })
                            })
                            .then(res => res.json())
                            .then(result => {
                                console.log('Verification response:', result);
                                
                                if (result.success) {
                                    // Show SweetAlert popup
                                    Swal.fire({
                                        icon: 'success',
                                        title: '🎉 Payment Successful!',
                                        html: `
                                            <p style="font-size: 16px; margin: 15px 0;">Your subscription has been activated!</p>
                                            <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 20px; border-radius: 15px; color: white; margin: 20px 0;">
                                                <p style="margin: 5px 0;"><strong>Plan:</strong> ${planName}</p>
                                                <p style="margin: 5px 0;"><strong>Amount:</strong> ₹${price}</p>
                                            </div>
                                            <p style="color: #28a745; font-weight: 600;">✅ Telegram notification sent!</p>
                                        `,
                                        confirmButtonText: 'Go to Dashboard',
                                        confirmButtonColor: '#667eea',
                                        allowOutsideClick: false,
                                        timer: 5000,
                                        timerProgressBar: true
                                    }).then(() => {
                                        // Redirect to dashboard where session toast will also show
                                        window.location.href = 'dashboard.php';
                                    });
                                } else {
                                    showError(result.message || 'Payment verification failed');
                                }
                            })
                            .catch(error => {
                                console.error('Verification error:', error);
                                showError('Payment verification failed. Please contact support.');
                            });
                        },
                        "prefill": {
                            "name": "<?php echo htmlspecialchars($_SESSION['user_name'] ?? ''); ?>",
                            "email": "<?php echo htmlspecialchars($_SESSION['user_email'] ?? ''); ?>"
                        },
                        "theme": {
                            "color": "#667eea"
                        }
                    };
                    var rzp = new Razorpay(options);
                    rzp.open();
                } else {
                    showError(data.message || 'Failed to create order');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showError('Something went wrong. Please try again.');
            });
        }
    </script>
    
    <?php include 'includes/chat-widget.php'; ?>
</body>
</html>
