<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/models/User.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$userModel = new User();
$user = $userModel->getUserById($_SESSION['user_id']);

$success = '';
$error = '';

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_profile'])) {
        $name = trim($_POST['name'] ?? '');
        $telegramNumber = trim($_POST['telegram_number'] ?? '');
        
        if (empty($name)) {
            $error = 'Name is required';
        } else {
            $stmt = $userModel->db->prepare("UPDATE users SET name = ?, telegram_number = ? WHERE id = ?");
            if ($stmt->execute([$name, $telegramNumber, $_SESSION['user_id']])) {
                $_SESSION['profile_success'] = 'Profile updated successfully!';
                header('Location: profile.php');
                exit;
            } else {
                $error = 'Failed to update profile';
            }
        }
    }
    
    if (isset($_POST['change_password'])) {
        $currentPassword = $_POST['current_password'] ?? '';
        $newPassword = $_POST['new_password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        
        if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
            $error = 'All password fields are required';
        } elseif ($newPassword !== $confirmPassword) {
            $error = 'New passwords do not match';
        } elseif (strlen($newPassword) < 6) {
            $error = 'Password must be at least 6 characters';
        } elseif (!$userModel->verifyPassword($_SESSION['user_id'], $currentPassword)) {
            $error = 'Current password is incorrect';
        } else {
            if ($userModel->updatePassword($_SESSION['user_id'], $newPassword)) {
                $_SESSION['profile_success'] = 'Password changed successfully!';
                header('Location: profile.php');
                exit;
            } else {
                $error = 'Failed to change password';
            }
        }
    }
}

// Check for success message from redirect
if (isset($_SESSION['profile_success'])) {
    $success = $_SESSION['profile_success'];
    unset($_SESSION['profile_success']);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Profile - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    <style>
        .profile-header {
            background: linear-gradient(135deg, rgba(102, 126, 234, 0.15) 0%, rgba(118, 75, 162, 0.15) 100%);
            border: 2px solid rgba(102, 126, 234, 0.3);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
        }
        
        .profile-avatar {
            background: var(--primary-gradient);
            width: 100px;
            height: 100px;
            border-radius: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 48px;
            color: white;
            box-shadow: 0 10px 30px rgba(102, 126, 234, 0.4);
            font-weight: 700;
        }
        
        .info-badge {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 8px 16px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 12px;
            font-size: 13px;
            font-weight: 500;
        }
        
        .section-card {
            background: var(--glass-bg);
            backdrop-filter: blur(20px);
            border: 1px solid var(--glass-border);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 25px;
        }
        
        .section-header {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid rgba(255, 255, 255, 0.1);
        }
        
        .section-icon {
            background: var(--primary-gradient);
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            display: block;
            font-weight: 600;
            margin-bottom: 8px;
            font-size: 14px;
            color: var(--text-light);
        }
        
        .form-control {
            width: 100%;
            padding: 14px 18px;
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 12px;
            color: var(--text-light);
            font-size: 15px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--neon-blue);
            background: rgba(255, 255, 255, 0.08);
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .form-control:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        .form-hint {
            display: block;
            margin-top: 6px;
            font-size: 12px;
            opacity: 0.7;
            line-height: 1.5;
        }
        
        .telegram-guide {
            background: linear-gradient(135deg, rgba(0, 136, 204, 0.1) 0%, rgba(0, 170, 255, 0.1) 100%);
            border: 1px solid rgba(0, 136, 204, 0.3);
            border-radius: 15px;
            padding: 20px;
            margin-top: 15px;
        }
        
        .telegram-step {
            display: flex;
            align-items: flex-start;
            gap: 12px;
            margin-bottom: 12px;
            padding: 12px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 10px;
        }
        
        .step-number {
            background: var(--primary-gradient);
            width: 28px;
            height: 28px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 14px;
            flex-shrink: 0;
        }
        
        .btn-save {
            background: var(--primary-gradient);
            border: none;
            padding: 14px 32px;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            font-size: 15px;
            cursor: pointer;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .btn-save:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(102, 126, 234, 0.4);
        }
        
        .security-info {
            background: rgba(255, 193, 7, 0.1);
            border: 1px solid rgba(255, 193, 7, 0.3);
            border-radius: 12px;
            padding: 15px;
            margin-bottom: 20px;
        }
        
        @media (max-width: 768px) {
            .profile-header {
                padding: 20px;
            }
            
            .profile-avatar {
                width: 80px;
                height: 80px;
                font-size: 36px;
            }
            
            .section-card {
                padding: 20px;
            }
        }
    </style>
</head>
<body>
    <?php include 'includes/user-navbar.php'; ?>

    <div class="container mt-4">
        <!-- Profile Header -->
        <div class="profile-header" data-aos="fade-up">
            <div style="display: flex; align-items: center; gap: 25px; flex-wrap: wrap;">
                <div class="profile-avatar">
                    <?php echo strtoupper(substr($user['name'], 0, 1)); ?>
                </div>
                <div style="flex: 1; min-width: 250px;">
                    <h2 class="gradient-text" style="margin: 0 0 10px 0; font-size: 28px;">
                        <?php echo htmlspecialchars($user['name']); ?>
                    </h2>
                    <p style="margin: 0 0 15px 0; opacity: 0.8; font-size: 15px;">
                        📧 <?php echo htmlspecialchars($user['email']); ?>
                    </p>
                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <span class="info-badge">
                            <span>👤</span>
                            <span><?php echo ucfirst($user['role']); ?></span>
                        </span>
                        <span class="info-badge" style="<?php echo $user['status'] === 'active' ? 'background: rgba(74, 222, 128, 0.2); color: #4ade80;' : 'background: rgba(245, 87, 108, 0.2); color: #f5576c;'; ?>">
                            <span><?php echo $user['status'] === 'active' ? '✅' : '❌'; ?></span>
                            <span><?php echo ucfirst($user['status']); ?></span>
                        </span>
                        <span class="info-badge">
                            <span>📅</span>
                            <span>Member since <?php echo date('M Y', strtotime($user['created_at'])); ?></span>
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <?php if ($success): ?>
            <div class="alert alert-success" data-aos="fade-down" style="margin-bottom: 20px;">
                ✅ <?php echo htmlspecialchars($success); ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-danger" data-aos="fade-down" style="margin-bottom: 20px;">
                ❌ <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <div class="row">
            <!-- Left Column - Profile Info -->
            <div class="col-md-6">
                <!-- Profile Information -->
                <div class="section-card" data-aos="fade-right">
                    <div class="section-header">
                        <div class="section-icon">👤</div>
                        <h3 class="gradient-text" style="margin: 0; font-size: 20px;">Profile Information</h3>
                    </div>

                    <form method="POST" action="">
                        <div class="form-group">
                            <label class="form-label">📛 Full Name</label>
                            <input type="text" name="name" class="form-control" value="<?php echo htmlspecialchars($user['name']); ?>" required>
                        </div>

                        <div class="form-group">
                            <label class="form-label">📧 Email Address</label>
                            <input type="email" class="form-control" value="<?php echo htmlspecialchars($user['email']); ?>" disabled>
                            <span class="form-hint">🔒 Email cannot be changed for security reasons</span>
                        </div>

                        <div class="form-group">
                            <label class="form-label">📱 Telegram Chat ID (Optional)</label>
                            <input type="text" name="telegram_number" class="form-control" 
                                   value="<?php echo htmlspecialchars($user['telegram_number'] ?? ''); ?>" 
                                   placeholder="e.g., 6034515846">
                            <span class="form-hint">
                                💡 Enter your Telegram Chat ID to receive instant notifications
                            </span>
                        </div>

                        <!-- Telegram Setup Guide -->
                        <div class="telegram-guide">
                            <h4 style="margin: 0 0 15px 0; font-size: 16px; color: #0088cc;">
                                📱 How to Get Your Telegram Chat ID
                            </h4>
                            
                            <div class="telegram-step">
                                <div class="step-number">1</div>
                                <div>
                                    <strong>Start the Bot</strong><br>
                                    <span style="opacity: 0.8; font-size: 13px;">
                                        Open Telegram and send <code style="background: rgba(0,0,0,0.3); padding: 2px 6px; border-radius: 4px;">/start</code> to 
                                        <a href="https://t.me/Informnotify_bot" target="_blank" style="color: #0088cc;">@Informnotify_bot</a>
                                    </span>
                                </div>
                            </div>

                            <div class="telegram-step">
                                <div class="step-number">2</div>
                                <div>
                                    <strong>Get Your Chat ID</strong><br>
                                    <span style="opacity: 0.8; font-size: 13px;">
                                        Search for <a href="https://t.me/userinfobot" target="_blank" style="color: #0088cc;">@userinfobot</a> 
                                        and send any message to get your Chat ID
                                    </span>
                                </div>
                            </div>

                            <div class="telegram-step">
                                <div class="step-number">3</div>
                                <div>
                                    <strong>Copy & Paste</strong><br>
                                    <span style="opacity: 0.8; font-size: 13px;">
                                        Copy the Chat ID number and paste it in the field above
                                    </span>
                                </div>
                            </div>

                            <div style="background: rgba(74, 222, 128, 0.1); padding: 12px; border-radius: 8px; margin-top: 12px;">
                                <span style="color: #4ade80; font-size: 13px;">
                                    ✅ Once set up, you'll receive instant notifications for payments, searches, and more!
                                </span>
                            </div>
                        </div>

                        <button type="submit" name="update_profile" class="btn-save" style="margin-top: 20px;">
                            <span>💾</span>
                            <span>Save Profile</span>
                        </button>
                    </form>
                </div>
            </div>

            <!-- Right Column - Security -->
            <div class="col-md-6">
                <!-- Change Password -->
                <div class="section-card" data-aos="fade-left">
                    <div class="section-header">
                        <div class="section-icon">🔒</div>
                        <h3 class="gradient-text" style="margin: 0; font-size: 20px;">Change Password</h3>
                    </div>

                    <div class="security-info">
                        <strong>🛡️ Security Tips:</strong>
                        <ul style="margin: 10px 0 0 0; padding-left: 20px; font-size: 13px; opacity: 0.9;">
                            <li>Use at least 6 characters</li>
                            <li>Mix letters, numbers, and symbols</li>
                            <li>Don't reuse passwords from other sites</li>
                        </ul>
                    </div>

                    <form method="POST" action="">
                        <div class="form-group">
                            <label class="form-label">🔑 Current Password</label>
                            <input type="password" name="current_password" class="form-control" placeholder="Enter your current password" required>
                        </div>

                        <div class="form-group">
                            <label class="form-label">🆕 New Password</label>
                            <input type="password" name="new_password" class="form-control" minlength="6" placeholder="Minimum 6 characters" required>
                        </div>

                        <div class="form-group">
                            <label class="form-label">✅ Confirm New Password</label>
                            <input type="password" name="confirm_password" class="form-control" minlength="6" placeholder="Re-enter new password" required>
                        </div>

                        <button type="submit" name="change_password" class="btn-save">
                            <span>🔐</span>
                            <span>Change Password</span>
                        </button>
                    </form>
                </div>

                <!-- Account Info -->
                <?php if (isset($user['last_login'])): ?>
                <div class="section-card" data-aos="fade-left" data-aos-delay="100">
                    <div class="section-header">
                        <div class="section-icon">📊</div>
                        <h3 class="gradient-text" style="margin: 0; font-size: 20px;">Account Activity</h3>
                    </div>

                    <div style="display: grid; gap: 15px;">
                        <div style="background: rgba(255, 255, 255, 0.05); padding: 15px; border-radius: 12px;">
                            <p style="margin: 0; opacity: 0.7; font-size: 12px; text-transform: uppercase; letter-spacing: 0.5px;">Last Login</p>
                            <p style="margin: 5px 0 0 0; font-weight: 600; font-size: 15px;">
                                🕐 <?php echo $user['last_login'] ? date('d M Y, h:i A', strtotime($user['last_login'])) : 'Never'; ?>
                            </p>
                        </div>

                        <div style="background: rgba(255, 255, 255, 0.05); padding: 15px; border-radius: 12px;">
                            <p style="margin: 0; opacity: 0.7; font-size: 12px; text-transform: uppercase; letter-spacing: 0.5px;">IP Address</p>
                            <p style="margin: 5px 0 0 0; font-weight: 600; font-size: 15px; font-family: monospace;">
                                🌐 <?php echo htmlspecialchars($user['last_ip'] ?? 'Unknown'); ?>
                            </p>
                        </div>

                        <div style="background: rgba(255, 255, 255, 0.05); padding: 15px; border-radius: 12px;">
                            <p style="margin: 0; opacity: 0.7; font-size: 12px; text-transform: uppercase; letter-spacing: 0.5px;">Device</p>
                            <p style="margin: 5px 0 0 0; font-size: 13px; word-break: break-word; opacity: 0.9;">
                                💻 <?php echo htmlspecialchars(substr($user['last_device'] ?? 'Unknown', 0, 60)); ?>
                            </p>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <!-- Chat Widget -->
    <?php include 'includes/chat-widget.php'; ?>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="assets/js/toast.js"></script>
    <script>
        AOS.init({
            duration: 1000,
            once: true
        });

        <?php if ($success): ?>
        showSuccess('<?php echo addslashes($success); ?>');
        <?php endif; ?>

        <?php if ($error): ?>
        showError('<?php echo addslashes($error); ?>');
        <?php endif; ?>
    </script>
</body>
</html>
