<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/models/User.php';
require_once __DIR__ . '/../app/controllers/Notification.php';

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';

    if (empty($name) || empty($email) || empty($password)) {
        $error = 'All fields are required';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Invalid email format';
    } elseif (strlen($password) < 6) {
        $error = 'Password must be at least 6 characters';
    } elseif ($password !== $confirm_password) {
        $error = 'Passwords do not match';
    } else {
        $userModel = new User();

        // Check if email already exists
        $stmt = $userModel->db->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);

        if ($stmt->fetch()) {
            $error = '❌ This email is already registered. Please use a different email or <a href="login.php" style="color: var(--neon-blue);">login here</a>.';
        } else {
            $userId = $userModel->register($name, $email, $password);

            if ($userId) {
                $user = $userModel->getUserById($userId);

                // Send notifications
                $notification = new Notification();
                $notification->notifyNewUser($user);

                $success = '🎉 Registration successful! You have received 10 FREE searches to get started. Please login.';
                header('refresh:3;url=login.php');
            } else {
                $error = '❌ Registration failed. Please try again or contact support.';
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap"
        rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
</head>

<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="container">
            <a href="index.php" class="navbar-brand">⚡ Gravity</a>
            <ul class="navbar-menu">
                <li><a href="index.php">Home</a></li>
                <li><a href="login.php" class="btn btn-outline">Login</a></li>
            </ul>
        </div>
    </nav>

    <!-- Registration Form -->
    <div class="container" style="max-width: 500px; margin-top: 80px;">
        <div class="glass-card" data-aos="fade-up">
            <h2 class="text-center gradient-text mb-3">Create Account</h2>
            <div style="text-align: center; margin-bottom: 20px;">
                <span class="badge badge-success" style="font-size: 14px; padding: 8px 16px;">
                    🎁 Get 10 FREE Searches - No Credit Card Required
                </span>
            </div>
            <p class="text-center mb-4" style="opacity: 0.8;">Join thousands of users today</p>

            <?php if ($error): ?>
                <div class="alert alert-danger"><?php echo $error; ?></div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
            <?php endif; ?>

            <form method="POST" action="">
                <div class="mb-2">
                    <label class="form-label">Full Name</label>
                    <input type="text" name="name" class="form-control" placeholder="Enter your full name" required
                        value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>">
                </div>

                <div class="mb-2">
                    <label class="form-label">Email Address</label>
                    <input type="email" name="email" class="form-control" placeholder="Enter your email" required
                        value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                </div>

                <div class="mb-2">
                    <label class="form-label">Password</label>
                    <input type="password" name="password" class="form-control" placeholder="Create a password"
                        required>
                </div>

                <div class="mb-3">
                    <label class="form-label">Confirm Password</label>
                    <input type="password" name="confirm_password" class="form-control"
                        placeholder="Confirm your password" required>
                </div>

                <button type="submit" class="btn btn-primary" style="width: 100%;">Register Now</button>
            </form>

            <p class="text-center mt-3" style="opacity: 0.8;">
                Already have an account? <a href="login.php" style="color: var(--neon-blue);">Login here</a>
            </p>
        </div>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="assets/js/toast.js"></script>
    <script>
        AOS.init({
            duration: 1000,
            once: true
        });

        <?php if ($success): ?>
            showSuccess('<?php echo addslashes($success); ?>');
        <?php endif; ?>

        <?php if ($error): ?>
            showError('<?php echo addslashes($error); ?>');
        <?php endif; ?>
    </script>
</body>

</html>