<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/models/User.php';
require_once __DIR__ . '/../app/models/Subscription.php';
require_once __DIR__ . '/../app/models/Search.php';
require_once __DIR__ . '/../app/controllers/Notification.php';

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'user') {
    header('Location: login.php');
    exit;
}

$searchType = $_GET['type'] ?? 'mobile';
if (!in_array($searchType, ['mobile', 'aadhaar'])) {
    $searchType = 'mobile';
}

$userModel = new User();
$subscriptionModel = new Subscription();
$searchModel = new Search();
$notification = new Notification();

$user = $userModel->getUserById($_SESSION['user_id']);
$activeSubscription = $subscriptionModel->getActiveSubscription($_SESSION['user_id']);
$remainingSearches = $subscriptionModel->getRemainingSearches($_SESSION['user_id']);

$error = '';
$result = null;
$source = '';
$isSearching = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $isSearching = true;
    $queryValue = trim($_POST['query'] ?? '');

    if (empty($queryValue)) {
        $error = 'Please enter a ' . $searchType . ' number';
    } elseif (!$activeSubscription) {
        $error = 'No active subscription. Please purchase a plan first.';
    } elseif ($remainingSearches <= 0) {
        $error = 'Search limit exhausted. Please upgrade your plan.';
    } else {
        // Check cache first
        $cachedSearch = $searchModel->getCachedSearch($searchType, $queryValue);
        
        if ($cachedSearch) {
            $result = json_decode($cachedSearch['json_response'], true);
            $source = 'cache';
            $subscriptionModel->incrementUsedSearches($activeSubscription['id']);
            $remainingSearches--;
            
            // Async notification
            register_shutdown_function(function() use ($notification, $user, $searchType, $queryValue, $source) {
                try {
                    $notification->notifyNewSearch($user, $searchType, $queryValue, $source);
                } catch (Exception $e) {
                    error_log("Notification error: " . $e->getMessage());
                }
            });
        } else {
            // Call API
            $apiUrl = $searchType === 'mobile' ? MOBILE_LOOKUP_API_URL : AADHAAR_LOOKUP_API_URL;
            $apiUrl .= urlencode($queryValue);
            
            try {
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $apiUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 10);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $curlError = curl_error($ch);
                curl_close($ch);
                
                if ($httpCode === 200 && $response) {
                    $result = json_decode($response, true);
                    
                    if ($result) {
                        $source = 'api';
                        
                        try {
                            $saved = $searchModel->saveSearch($_SESSION['user_id'], $searchType, $queryValue, $response, $source);
                        } catch (Exception $e) {
                            error_log("Error saving search: " . $e->getMessage());
                        }
                        
                        $subscriptionModel->incrementUsedSearches($activeSubscription['id']);
                        $remainingSearches--;
                        
                        // Async notification
                        register_shutdown_function(function() use ($notification, $user, $searchType, $queryValue, $source) {
                            try {
                                $notification->notifyNewSearch($user, $searchType, $queryValue, $source);
                            } catch (Exception $e) {
                                error_log("Notification error: " . $e->getMessage());
                            }
                        });
                    } else {
                        $error = 'Invalid response from API - Could not decode JSON. Response: ' . substr($response, 0, 200);
                    }
                } else {
                    $error = "API Error - HTTP Code: $httpCode" . ($curlError ? " - $curlError" : "") . " - Response: " . substr($response, 0, 100);
                }
            } catch (Exception $e) {
                $error = 'Exception: ' . $e->getMessage();
            }
        }
    }
    $isSearching = false;
}

// Debug output (temporary - remove after fixing)
if (isset($_POST['query']) && !$result && !$error) {
    $error = 'DEBUG: Search processed but no result or error set. Check logic flow.';
}

// Helper function to check if array is list of results (numeric keys)
function isResultsList($array) {
    if (!is_array($array)) return false;
    $keys = array_keys($array);
    return isset($keys[0]) && is_numeric($keys[0]);
}

// Helper function to display single field
function displayField($key, $value) {
    // Skip unwanted fields
    $skipFields = ['valid_until', 'validuntil', 'success', 'credit', 'credits', 'api_credit'];
    if (in_array(strtolower($key), $skipFields)) {
        return '';
    }
    
    // Skip empty values
    if (empty($value) && $value !== '0') {
        return '';
    }
    
    // Format label
    $label = ucwords(str_replace(['_', '-'], ' ', $key));
    
    // Return formatted row
    return '<div style="padding: 12px; margin: 8px 0; background: rgba(255,255,255,0.05); border-left: 3px solid #667eea; border-radius: 8px;">
                <div style="color: rgba(255,255,255,0.6); font-size: 12px; margin-bottom: 4px;">' . htmlspecialchars($label) . '</div>
                <div style="color: #fff; font-size: 15px; font-weight: 500; word-break: break-word;">' . htmlspecialchars($value) . '</div>
            </div>';
}

// Flatten nested array to simple key-value pairs
function flattenArray($array, $prefix = '') {
    $result = [];
    
    foreach ($array as $key => $value) {
        $newKey = $prefix ? $prefix . ' ' . $key : $key;
        
        if (is_array($value)) {
            // If it's a list (numeric keys), skip it
            if (isResultsList($value)) {
                continue;
            }
            // Recursively flatten
            $result = array_merge($result, flattenArray($value, $newKey));
        } else {
            $result[$newKey] = $value;
        }
    }
    
    return $result;
}

// Helper function to display single result block
function displayResultBlock($data, $index = null) {
    $output = '<div style="background: rgba(255,255,255,0.03); padding: 20px; border-radius: 15px; border: 1px solid rgba(255,255,255,0.1); margin-bottom: 20px;">';
    
    if ($index !== null) {
        $output .= '<div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); display: inline-block; padding: 6px 16px; border-radius: 20px; margin-bottom: 15px; font-size: 13px; font-weight: 600;">
                        Result #' . ($index + 1) . '
                    </div>';
    }
    
    // Flatten the data to get all fields
    $flatData = flattenArray($data);
    
    // Display all fields
    foreach ($flatData as $key => $value) {
        $output .= displayField($key, $value);
    }
    
    // If no fields displayed, show raw data for debugging
    if (count($flatData) == 0) {
        $output .= '<div style="color: #ff6b6b; padding: 10px;">No displayable data found</div>';
        $output .= '<pre style="color: #fff; font-size: 12px; overflow: auto; max-height: 300px;">' . htmlspecialchars(print_r($data, true)) . '</pre>';
    }
    
    $output .= '</div>';
    return $output;
}
?>



<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo ucfirst($searchType); ?> Search - <?php echo SITE_NAME; ?></title>
    
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link rel="preconnect" href="https://unpkg.com">
    
    <link rel="stylesheet" href="assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    <style>
        /* Prevent blank page during reload */
        body {
            background: linear-gradient(135deg, #0d1117 0%, #1a1f2e 100%);
            min-height: 100vh;
        }
        
        /* Page-level loading screen */
        #pageLoader {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, #0d1117 0%, #1a1f2e 100%);
            z-index: 99999;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: opacity 0.3s ease;
        }
        
        #pageLoader.hidden {
            opacity: 0;
            pointer-events: none;
        }
        
        /* Hide page content while loading */
        #pageContent {
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        #pageContent.visible {
            opacity: 1;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        @keyframes progress {
            0% { transform: translateX(-100%); }
            100% { transform: translateX(100%); }
        }
    </style>
</head>
<body>
    <!-- Page Loader - Shows during page load -->
    <div id="pageLoader">
        <div style="text-align: center;">
            <div style="width: 80px; height: 80px; margin: 0 auto 30px; border: 5px solid rgba(102, 126, 234, 0.2); border-top: 5px solid #667eea; border-radius: 50%; animation: spin 0.8s linear infinite;"></div>
            <p style="color: white; font-size: 24px; font-weight: 700; margin: 0 0 10px 0;">🔍 Searching...</p>
            <p style="color: rgba(255,255,255,0.7); font-size: 16px; margin: 0;">Please wait while we fetch the results</p>
            <div style="margin-top: 30px;">
                <div style="width: 200px; height: 4px; background: rgba(255,255,255,0.1); border-radius: 2px; margin: 0 auto; overflow: hidden;">
                    <div style="width: 100%; height: 100%; background: var(--primary-gradient); animation: progress 1.5s ease-in-out infinite;"></div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // Hide page loader and show content when ready
        window.addEventListener('load', function() {
            setTimeout(function() {
                document.getElementById('pageLoader').classList.add('hidden');
                document.getElementById('pageContent').classList.add('visible');
                setTimeout(function() {
                    document.getElementById('pageLoader').style.display = 'none';
                }, 300);
            }, 100);
        });
        
        // Show page loader if coming from search submission
        if (sessionStorage.getItem('searchLoading') === 'true') {
            document.getElementById('pageLoader').style.display = 'flex';
        }
    </script>
    
    <!-- Page Content Wrapper -->
    <div id="pageContent">
    <?php include 'includes/user-navbar.php'; ?>

    <div class="container mt-4">
        <!-- Search Type Tabs -->
        <div class="text-center mb-4" data-aos="fade-down">
            <a href="search.php?type=mobile" class="btn <?php echo $searchType === 'mobile' ? 'btn-primary' : 'btn-outline'; ?>" style="margin: 0 10px;">
                📱 Mobile Search
            </a>
            <a href="search.php?type=aadhaar" class="btn <?php echo $searchType === 'aadhaar' ? 'btn-primary' : 'btn-outline'; ?>" style="margin: 0 10px;">
                🆔 Aadhaar Search
            </a>
        </div>

        <!-- Search Form -->
        <div class="row">
            <div class="col-md-6" style="margin: 0 auto;">
                <div class="glass-card" data-aos="fade-up" style="position: relative;">
                    <h2 class="text-center gradient-text mb-3"><?php echo ucfirst($searchType); ?> Lookup</h2>
                    
                    <?php if ($activeSubscription): ?>
                        <div class="alert alert-info mb-3">
                            <strong>Remaining Searches:</strong> <?php echo $remainingSearches; ?> / <?php echo $activeSubscription['total_search_limit']; ?>
                        </div>
                    <?php endif; ?>

                    <?php if ($error): ?>
                        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
                    <?php endif; ?>

                    <form method="POST" action="" id="searchForm">
                        <div class="mb-3">
                            <label class="form-label">Enter <?php echo ucfirst($searchType); ?> Number</label>
                            <input type="text" name="query" id="queryInput" class="form-control" 
                                   placeholder="<?php echo $searchType === 'mobile' ? 'e.g., 9876543210' : 'e.g., 1234 5678 9012'; ?>" 
                                   required 
                                   value="<?php echo htmlspecialchars($_POST['query'] ?? ''); ?>">
                        </div>

                        <button type="submit" class="btn btn-primary" id="searchBtn" style="width: 100%;">
                            🔍 Search Now
                        </button>
                    </form>
                    
                    <!-- Loading Overlay - Full Page -->
                    <div id="loadingOverlay" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(13, 17, 23, 0.98); z-index: 9999; backdrop-filter: blur(10px);">
                        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); text-align: center;">
                            <div class="spinner" style="width: 80px; height: 80px; margin: 0 auto 30px; border: 5px solid rgba(102, 126, 234, 0.2); border-top: 5px solid #667eea; border-radius: 50%; animation: spin 0.8s linear infinite;"></div>
                            <p style="color: white; font-size: 24px; font-weight: 700; margin: 0 0 10px 0;">🔍 Searching...</p>
                            <p style="color: rgba(255,255,255,0.7); font-size: 16px; margin: 0;">Please wait while we fetch the results</p>
                            <div style="margin-top: 30px;">
                                <div style="width: 200px; height: 4px; background: rgba(255,255,255,0.1); border-radius: 2px; margin: 0 auto; overflow: hidden;">
                                    <div style="width: 100%; height: 100%; background: var(--primary-gradient); animation: progress 1.5s ease-in-out infinite;"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Search Results -->
        <?php if ($result): ?>
            <div class="mt-4" data-aos="fade-up">
                <div class="glass-card">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid rgba(255,255,255,0.1);">
                        <h3 class="gradient-text" style="margin: 0;">📊 Search Results</h3>
                        <span class="badge <?php echo $source === 'cache' ? 'badge-success' : 'badge-warning'; ?>" style="font-size: 13px; padding: 8px 16px;">
                            <?php echo $source === 'cache' ? '⚡ CACHED' : '🌐 LIVE API'; ?>
                        </span>
                    </div>
                    
                    <!-- Results Display -->
                    <div style="max-width: 900px; margin: 0 auto;">
                        <?php
                        // Extract the actual data array from nested structure
                        $resultsToDisplay = [];
                        
                        // Check if result has nested data → data → array structure
                        if (isset($result['data']) && is_array($result['data'])) {
                            if (isset($result['data']['data']) && is_array($result['data']['data'])) {
                                // This is the actual results array
                                $resultsToDisplay = $result['data']['data'];
                            } else {
                                $resultsToDisplay = [$result['data']];
                            }
                        } else {
                            $resultsToDisplay = [$result];
                        }
                        
                        // Check if it's a list of results
                        if (isResultsList($resultsToDisplay)) {
                            // Multiple results - show count and separate blocks
                            $count = count($resultsToDisplay);
                            
                            if ($count > 1) {
                                echo '<div style="background: rgba(102, 126, 234, 0.1); padding: 12px 20px; border-radius: 10px; margin-bottom: 20px; border-left: 4px solid #667eea;">
                                        <strong>🔍 Found ' . $count . ' Results</strong>
                                      </div>';
                            }
                            
                            foreach ($resultsToDisplay as $index => $singleResult) {
                                echo displayResultBlock($singleResult, $count > 1 ? $index : null);
                            }
                        } else {
                            // Single result
                            echo displayResultBlock($resultsToDisplay[0] ?? $result);
                        }
                        ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="assets/js/toast.js"></script>
    <script>
        AOS.init({
            duration: 1000,
            once: true
        });

        // Check if we're loading from a search submission
        window.addEventListener('DOMContentLoaded', function() {
            if (sessionStorage.getItem('searchLoading') === 'true') {
                document.getElementById('loadingOverlay').style.display = 'block';
                // Clear the flag after showing
                sessionStorage.removeItem('searchLoading');
            }
        });

        // Form submission handler
        document.getElementById('searchForm').addEventListener('submit', function(e) {
            // Set loading flag in sessionStorage
            sessionStorage.setItem('searchLoading', 'true');
            
            // Show loading overlay
            document.getElementById('loadingOverlay').style.display = 'block';
            document.getElementById('searchBtn').disabled = true;
            document.getElementById('searchBtn').innerHTML = '⏳ Searching...';
            
            // Let form submit normally
            return true;
        });

        <?php if ($result): ?>
        // Clear loading state when results are shown
        sessionStorage.removeItem('searchLoading');
        showSuccess('Search successful! Results loaded from <?php echo $source; ?>');
        <?php endif; ?>

        <?php if ($error): ?>
        // Clear loading state on error
        sessionStorage.removeItem('searchLoading');
        showError('<?php echo addslashes($error); ?>');
        <?php endif; ?>
    </script>
    
    </div> <!-- End pageContent -->
    
    <?php include 'includes/chat-widget.php'; ?>
</body>
</html>
