<?php
session_start();
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/models/Search.php';
require_once __DIR__ . '/../app/models/Database.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$searchId = $_GET['id'] ?? 0;
$searchModel = new Search();
$search = $searchModel->getSearchById($searchId);

if (!$search || $search['user_id'] != $_SESSION['user_id']) {
    header('Location: history.php');
    exit;
}

// Check view permissions
$db = Database::getInstance()->getConnection();
$stmt = $db->query("SELECT setting_value FROM search_settings WHERE setting_key = 'allow_repeat_views'");
$allowRepeatViews = $stmt->fetch()['setting_value'] ?? '1';

$canView = true;
$lockMessage = '';

// Check if admin has locked this search
if ($search['admin_locked']) {
    $canView = false;
    $lockMessage = '🔒 This search has been locked by admin';
}
// Check if repeat views are disabled and user has already viewed
elseif ($allowRepeatViews == '0' && $search['view_count'] > 0) {
    $canView = false;
    $lockMessage = '🔒 You have already viewed this search. Repeat views are disabled.';
}
// Check individual search lock
elseif (!$search['can_view_again'] && $search['view_count'] > 0) {
    $canView = false;
    $lockMessage = '🔒 This search can only be viewed once';
}

// Increment view count if allowed
if ($canView) {
    $stmt = $db->prepare("UPDATE searches SET view_count = view_count + 1 WHERE id = ?");
    $stmt->execute([$searchId]);
}

// Parse JSON response
$data = json_decode($search['json_response'], true);

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv' && $canView) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="search_' . $searchId . '_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // Add headers
    fputcsv($output, ['Field', 'Value']);
    
    // Add search info
    fputcsv($output, ['Search Type', strtoupper($search['search_type'])]);
    fputcsv($output, ['Query', $search['query_value']]);
    fputcsv($output, ['Date', date('d M Y H:i', strtotime($search['created_at']))]);
    fputcsv($output, ['Source', strtoupper($search['source'])]);
    fputcsv($output, ['']);
    
    // Add data
    if (is_array($data)) {
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                fputcsv($output, [$key, json_encode($value)]);
            } else {
                fputcsv($output, [$key, $value]);
            }
        }
    }
    
    fclose($output);
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Search Results - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/user-navbar.php'; ?>

    <div class="container mt-4">
        <!-- Header -->
        <div class="glass-card mb-4" data-aos="fade-up">
            <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 15px;">
                <div>
                    <h2 class="gradient-text" style="margin-bottom: 10px;">
                        <?php echo $search['search_type'] === 'mobile' ? '📱' : '🆔'; ?> 
                        <?php echo ucfirst($search['search_type']); ?> Search Results
                    </h2>
                    <p style="opacity: 0.8; margin: 0;">
                        <strong>Query:</strong> <?php echo htmlspecialchars($search['query_value']); ?> | 
                        <strong>Date:</strong> <?php echo date('d M Y, h:i A', strtotime($search['created_at'])); ?> |
                        <strong>Source:</strong> <span class="badge <?php echo $search['source'] === 'cache' ? 'badge-success' : 'badge-warning'; ?>">
                            <?php echo strtoupper($search['source']); ?>
                        </span>
                    </p>
                </div>
                <div style="display: flex; gap: 10px;">
                    <?php if ($canView): ?>
                        <a href="?id=<?php echo $searchId; ?>&export=csv" class="btn btn-success">
                            📥 Export CSV
                        </a>
                    <?php endif; ?>
                    <a href="history.php" class="btn btn-outline">← Back to History</a>
                </div>
            </div>
        </div>

        <?php if (!$canView): ?>
            <!-- Locked Message -->
            <div class="glass-card text-center" data-aos="fade-up" style="background: rgba(255, 193, 7, 0.1); border-color: rgba(255, 193, 7, 0.3);">
                <div style="font-size: 64px; margin-bottom: 20px;">🔒</div>
                <h3 style="color: #ffc107; margin-bottom: 15px;"><?php echo $lockMessage; ?></h3>
                <p style="opacity: 0.8; margin-bottom: 20px;">
                    <?php if ($search['admin_locked']): ?>
                        This search has been restricted by the administrator.
                    <?php else: ?>
                        Contact admin if you need to view this search again.
                    <?php endif; ?>
                </p>
                <div style="background: rgba(255, 255, 255, 0.1); padding: 15px; border-radius: 10px; margin-top: 20px;">
                    <p style="margin: 0; opacity: 0.7;">
                        <strong>Views:</strong> <?php echo $search['view_count']; ?> time(s) | 
                        <strong>Last Viewed:</strong> <?php echo date('d M Y, h:i A', strtotime($search['created_at'])); ?>
                    </p>
                </div>
                <a href="history.php" class="btn btn-primary mt-3">Go to History</a>
            </div>
        <?php else: ?>
            <!-- Results Cards -->
            <?php 
            // Extract actual data from nested JSON
            $displayData = [];
            
            if (is_array($data)) {
                // Check if data has nested structure
                if (isset($data['data'])) {
                    $innerData = $data['data'];
                    
                    // If data.data is array with items
                    if (is_array($innerData) && isset($innerData['data']) && is_array($innerData['data'])) {
                        // Get first item from data.data array
                        if (count($innerData['data']) > 0) {
                            $displayData = $innerData['data'][0];
                        }
                    }
                    // If data.data is direct array
                    elseif (is_array($innerData) && count($innerData) > 0) {
                        $displayData = is_array($innerData[0]) ? $innerData[0] : $innerData;
                    }
                }
                
                // If no nested data found, use top level
                if (empty($displayData)) {
                    $displayData = $data;
                }
            }
            
            // Filter out meta fields
            $excludeFields = ['success', 'valid_until', 'data', 'credit', 'requested_number'];
            $cleanData = [];
            
            foreach ($displayData as $key => $value) {
                if (!in_array($key, $excludeFields) && !is_array($value) && !is_object($value)) {
                    $cleanData[$key] = $value;
                }
            }
            ?>
            
            <div class="row">
                <?php if (count($cleanData) > 0): ?>
                    <?php 
                    $cardIndex = 0;
                    foreach ($cleanData as $key => $value): 
                        $cardIndex++;
                    ?>
                        <div class="col-md-6 mb-3" data-aos="fade-up" data-aos-delay="<?php echo $cardIndex * 50; ?>">
                            <div class="glass-card" style="height: 100%;">
                                <div style="display: flex; align-items: flex-start; gap: 15px;">
                                    <div style="background: var(--primary-gradient); width: 50px; height: 50px; border-radius: 12px; display: flex; align-items: center; justify-content: center; flex-shrink: 0;">
                                        <span style="font-size: 24px;">
                                            <?php 
                                            $icons = [
                                                'name' => '👤', 
                                                'fname' => '👨', 
                                                'mobile' => '📱', 
                                                'phone' => '📱',
                                                'email' => '📧', 
                                                'address' => '📍', 
                                                'aadhaar' => '🆔', 
                                                'dob' => '🎂', 
                                                'gender' => '⚧', 
                                                'state' => '🗺️', 
                                                'city' => '🏙️', 
                                                'pincode' => '📮',
                                                'circle' => '🌐',
                                                'id' => '🔢'
                                            ];
                                            echo $icons[strtolower($key)] ?? '📄';
                                            ?>
                                        </span>
                                    </div>
                                    <div style="flex: 1; min-width: 0;">
                                        <h4 style="margin: 0 0 8px 0; font-size: 14px; opacity: 0.7; text-transform: uppercase; letter-spacing: 1px;">
                                            <?php echo htmlspecialchars(ucwords(str_replace('_', ' ', $key))); ?>
                                        </h4>
                                        <p style="margin: 0; font-size: 16px; font-weight: 600; word-wrap: break-word;">
                                            <?php echo htmlspecialchars($value); ?>
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="col-md-12">
                        <div class="glass-card text-center">
                            <p style="opacity: 0.7;">No data found in search results</p>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

            <!-- View Info -->
            <div class="glass-card mt-4" data-aos="fade-up" style="background: rgba(79, 172, 254, 0.1); border-color: rgba(79, 172, 254, 0.3);">
                <div style="text-align: center;">
                    <p style="margin: 0; opacity: 0.9;">
                        <strong>👁️ Views:</strong> <?php echo $search['view_count'] + 1; ?> time(s)
                        <?php if ($allowRepeatViews == '0'): ?>
                            | <span style="color: #ffc107;">⚠️ This is your only view</span>
                        <?php endif; ?>
                    </p>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Footer -->
    <footer class="footer">
        <div class="container">
            <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
        </div>
    </footer>

    <!-- Chat Widget -->
    <?php include 'includes/chat-widget.php'; ?>

    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="assets/js/toast.js"></script>
    <script>
        AOS.init({
            duration: 1000,
            once: true
        });
    </script>
</body>
</html>
