# Email OTP Setup Guide

## Problem
Email OTP wasn't working because:
1. ❌ Exposed credentials in config files
2. ❌ Using basic PHP mail() without SMTP
3. ❌ No error handling for mail failures

## Solution Implemented
✅ **SMTP Support** - Direct connection to Gmail/SMTP servers  
✅ **Secure Credentials** - Using .env file instead of config  
✅ **Fallback Support** - Falls back to PHP mail if SMTP fails  
✅ **Better Logging** - Detailed error logging

---

## Setup Instructions

### Step 1: Create .env File
Create a file named `.env` in your project root (same level as index.php):

```
EMAIL_MODE=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_ENCRYPTION=tls
MAIL_USERNAME=your-email@gmail.com
MAIL_PASSWORD=your-app-password

MAIL_FROM_EMAIL=your-email@gmail.com
MAIL_FROM_NAME=Advanced Mobile Info
```

### Step 2: Get Gmail App Password
**For Gmail users:**

1. Go to: https://myaccount.google.com/apppasswords
2. Select "Mail" and "Windows Computer"
3. Generate App Password (16 characters)
4. Copy the password (without spaces)
5. Paste in `.env` as `MAIL_PASSWORD`

**Note:** You must have **2-Factor Authentication** enabled first!

### Step 3: Load .env File
Update your `config/config.php` to load the .env file:

```php
<?php
// Load .env file
if (file_exists(__DIR__ . '/../.env')) {
    $lines = file(__DIR__ . '/../.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '#') === 0) continue;
        if (strpos($line, '=') === false) continue;
        
        [$key, $value] = explode('=', $line, 2);
        $key = trim($key);
        $value = trim($value);
        putenv("$key=$value");
    }
}

// Rest of your config...
?>
```

### Step 4: Test OTP System
1. Go to registration page
2. Enter your email
3. Check email inbox (might be in spam)
4. Verify OTP

---

## Troubleshooting

### "Connection refused"
- Check MAIL_HOST, MAIL_PORT, MAIL_ENCRYPTION are correct
- Verify internet connection is working

### "Authentication failed"
- Gmail App Password is incorrect or has spaces
- 2FA not enabled on Gmail account
- Regular Gmail password won't work (must be App Password)

### Email not arriving
- Check spam folder
- Verify MAIL_FROM_EMAIL is correct
- Check `error_log` file for details
- Falls back to PHP mail() if SMTP fails

### Check Logs
View detailed logs:
```bash
tail -f error_log  # Linux/Mac
type error_log     # Windows
```

---

## Alternative Email Services

You can use other SMTP services:

**SendGrid:**
```
MAIL_HOST=smtp.sendgrid.net
MAIL_PORT=587
MAIL_USERNAME=apikey
MAIL_PASSWORD=SG.xxxxx
```

**AWS SES:**
```
MAIL_HOST=email-smtp.region.amazonaws.com
MAIL_PORT=587
```

**Office 365:**
```
MAIL_HOST=smtp.office365.com
MAIL_PORT=587
```

---

## Security Best Practices

1. ✅ Never commit `.env` to version control
2. ✅ Add `.env` to `.gitignore`
3. ✅ Use app-specific passwords, not main account password
4. ✅ Rotate credentials regularly
5. ✅ Monitor error logs for failed attempts

---

## Files Modified

1. `config/email_config.php` - Now uses environment variables
2. `classes/OTPManager.php` - Added SMTP support
3. `.env` - New configuration file (add to .gitignore)

---

## Quick Test

```php
<?php
require 'config/config.php';
require 'classes/Database.php';
require 'classes/OTPManager.php';

$db = Database::getInstance();
$otp = new OTPManager($db);

$result = $otp->generateAndSendOTP('your-test-email@gmail.com', 'Test User');
echo json_encode($result);
?>
```

Run this file and check if email arrives within 5 minutes.
