# Email Settings - How It Works 📧

## Overview
The email system has been updated with a working email configuration panel in the Admin Dashboard. Here's how it works:

## System Architecture

### 1. **Configuration Files**
- **`config/email_config.php`**: Default configuration with environment variables
- **`admin/partials/email_settings.php`**: Admin panel form to update settings
- **`classes/OTPManager.php`**: Handles sending emails using saved settings

### 2. **Database Storage**
Email settings are stored in the `admin_settings` table with these keys:
```
- email_mode              → 'smtp' or 'php_mail'
- mail_host              → SMTP server (e.g., smtp.gmail.com)
- mail_port              → Port number (587, 465, etc.)
- mail_encryption        → 'tls', 'ssl', or 'none'
- mail_username          → Email account username
- mail_password_encrypted → Encrypted password (base64)
- mail_from_email        → Sender email address
- mail_from_name         → Sender display name (e.g., "Admin")
```

## How Email Sending Works

### Step 1: Load Settings
When OTPManager is initialized:
```php
$otpManager = new OTPManager($db);
```

It does this:
1. Loads default config from `email_config.php`
2. Queries database for `admin_settings` table
3. **Overrides** default values with database settings (if they exist)

### Step 2: Send Email
When sending an OTP:
```php
$otpManager->generateAndSendOTP($email, 'User Name');
```

The system:
1. Generates a 6-digit OTP
2. Stores it in `email_otps` table with 5-minute expiry
3. Calls `sendEmail()` which checks the `mode`:
   - If `mode = 'smtp'`: Uses `sendViaSMTP()` (recommended)
   - If `mode = 'php_mail'`: Uses PHP's mail() function

### Step 3: SMTP Connection
The SMTP sender:
1. Connects to the configured SMTP host on the specified port
2. Authenticates with username and **decrypted** password
3. Sends the email HTML
4. Closes the connection

## How to Configure

### Via Admin Dashboard
1. Go to **Admin Dashboard** → **Email Settings**
2. Fill in your SMTP details:
   - **Email Mode**: Choose SMTP (recommended) or PHP Mail
   - **SMTP Host**: smtp.gmail.com (for Gmail)
   - **SMTP Port**: 587 (for TLS) or 465 (for SSL)
   - **Encryption**: TLS or SSL
   - **Username**: Your email address
   - **Password**: Use Gmail App Password (not your actual password!)
   - **From Email**: The email that will appear as sender
   - **From Name**: The name that appears in emails

3. Click **Send Test Email** to verify it works

### For Gmail Users
⚠️ **Important**: Google doesn't allow regular passwords for SMTP

Steps:
1. Go to https://myaccount.google.com/apppasswords
2. Select **Mail** and **Windows Computer** (or your device)
3. Google will generate a 16-character app password
4. Copy and paste that into the Password field
5. Save and test

### Environment Variables (Alternative)
You can also set these in `.env` or your hosting environment:
```
EMAIL_MODE=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_ENCRYPTION=tls
MAIL_USERNAME=your-email@gmail.com
MAIL_PASSWORD=your-app-password
MAIL_FROM_EMAIL=your-email@gmail.com
MAIL_FROM_NAME=Advanced Mobile Info
```

## Troubleshooting

### Settings Not Saving?
✅ **Fixed in latest version** - The password field now saves correctly

### Emails Not Sending?
1. Check Admin Settings page - verify all fields are filled
2. Send a Test Email - see what error appears
3. Check error logs in `root/error_log`
4. Verify Gmail App Password is correct
5. Ensure port 587 or 465 is not blocked by firewall

### "SMTP credentials not configured"
This error appears when:
- Username or password is empty
- System will fall back to PHP mail() function
- **Solution**: Fill in all credential fields in Admin Settings

### "Connection Failed"
This usually means:
- Wrong SMTP host or port
- Firewall/network blocking the port
- Server doesn't allow outgoing SMTP connections
- **Solution**: Contact hosting provider about SMTP support

## Security Notes

✅ **Password Encryption**
- Passwords are base64 encoded in database (not truly encrypted)
- For production, consider using proper encryption library

⚠️ **Best Practices**
- Never share your Gmail password - use App Passwords only
- Store credentials securely in environment variables
- Regularly review authorized apps in Google Account
- Use strong, unique app passwords

## Technical Details

### Database Query Flow
```
Admin saves form
    ↓
Form POST to email_settings.php
    ↓
For each field: Check if exists in admin_settings
    ↓
If exists: UPDATE row | If not: INSERT row
    ↓
Password: Encrypt with base64 and save separately
    ↓
Show success message
```

### Email Sending Flow
```
User triggers action (register, OTP request, etc.)
    ↓
OTPManager initialized with $db
    ↓
loadDatabaseSettings() fetches settings from admin_settings
    ↓
generateAndSendOTP() creates OTP and calls sendEmail()
    ↓
sendEmail() checks 'mode'
    ↓
sendViaSMTP() or sendViaPhpMail()
    ↓
Email delivered or error logged
```

## Recent Changes

### Bug Fixed ✅
- **Password field**: Now correctly saves to database
- **Empty values**: Non-required fields no longer overwrite with empty values
- **Form validation**: Better error handling

## Files Involved
- `admin/partials/email_settings.php` - Form and save logic
- `classes/OTPManager.php` - Email sending logic
- `config/email_config.php` - Default configuration
- `setup_email_settings.php` - Initial setup script
- Database: `admin_settings` table

---

**Last Updated**: November 20, 2025
**Status**: ✅ Production Ready
