<?php
require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';
require_once '../classes/User.php';
require_once '../classes/OTPManager.php';

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

$userClass = new User($db);
$currentUser = $auth->getCurrentUser();

// Get full user data including email_verified status
$userDetails = $db->fetch(
    'SELECT * FROM users WHERE id = ?',
    [$currentUser['id']]
);

$message = '';
$saveSuccess = false;
$emailChanged = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');

    // Check if email changed
    $oldEmail = $userDetails['email'] ?? '';
    $emailChanged = ($email !== $oldEmail);

    if ($userClass->updateProfile($currentUser['id'], $name, $email)) {
        $message = '✅ Profile updated successfully';
        $saveSuccess = true;
        $_SESSION['user_name'] = $name;
        $_SESSION['user_email'] = $email;
        
        // If email changed, generate OTP and mark as unverified
        if ($emailChanged) {
            // Mark email as unverified
            $db->update(
                'users',
                ['email_verified' => 0, 'email_verified_at' => NULL],
                'id = ?',
                [$currentUser['id']]
            );
            
            // Delete old OTP if exists
            $db->query('DELETE FROM email_otps WHERE email = ?', [$oldEmail]);
            
            // Generate and send OTP to new email
            $otp = new OTPManager($db);
            $otpSent = $otp->generateAndSendOTP($email, $name);
            
            if ($otpSent) {
                $message = '✅ Profile updated! OTP sent to new email. Please verify it.';
            } else {
                $message = '✅ Profile updated but failed to send OTP. Please try again.';
            }
        }
        
        // Refresh user details
        $userDetails = $db->fetch(
            'SELECT * FROM users WHERE id = ?',
            [$currentUser['id']]
        );
    } else {
        $message = '❌ Failed to update profile. Please try again.';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>My Profile - <?php echo APP_NAME; ?></title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
<style>
body {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    min-height: 100vh;
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
    padding: 20px 0;
}
.container {
    max-width: 600px;
}
.card {
    border: none;
    border-radius: 16px;
    box-shadow: 0 8px 32px rgba(0,0,0,0.15);
}
.card-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 16px 16px 0 0 !important;
    padding: 24px;
}
.card-header h4 {
    margin: 0;
    font-weight: 600;
}
.form-control {
    border-radius: 8px;
    border: 1px solid #e0e0e0;
    padding: 12px 16px;
}
.form-control:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
}
.alert {
    border-radius: 12px;
    border: none;
}
.alert-success {
    background-color: #d4edda;
    color: #155724;
}
.alert-warning {
    background-color: #fff3cd;
    color: #856404;
}
.btn {
    border-radius: 8px;
    padding: 10px 24px;
    font-weight: 600;
}
.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    border-radius: 12px;
    font-size: 14px;
    font-weight: 600;
}
.status-verified {
    background-color: #d4edda;
    color: #155724;
}
.status-pending {
    background-color: #fff3cd;
    color: #856404;
}
</style>
</head>
<body>
<div class="container mt-5">
    <div class="card">
        <div class="card-header">
            <h4><i class="bi bi-person-circle"></i> My Profile</h4>
        </div>
        <div class="card-body p-4">
            <?php if ($message): ?>
                <div class="alert <?php echo $saveSuccess ? 'alert-success' : 'alert-danger'; ?>" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>

            <form method="post">
                <div class="mb-3">
                    <label class="form-label fw-semibold">Full Name</label>
                    <input type="text" name="name" class="form-control" value="<?php echo htmlspecialchars($userDetails['name'] ?? ''); ?>" required>
                </div>

                <div class="mb-3">
                    <label class="form-label fw-semibold">Email Address</label>
                    <div class="input-group">
                        <input type="email" name="email" class="form-control" value="<?php echo htmlspecialchars($userDetails['email'] ?? ''); ?>" required>
                        <span class="input-group-text" style="border: 1px solid #e0e0e0;">
                            <?php if ($userDetails && isset($userDetails['email_verified']) && $userDetails['email_verified']): ?>
                                <i class="bi bi-check-circle-fill text-success" title="Email Verified"></i>
                            <?php else: ?>
                                <i class="bi bi-exclamation-circle-fill text-warning" title="Email Not Verified"></i>
                            <?php endif; ?>
                        </span>
                    </div>
                </div>

                <!-- Email Verification Status -->
                <div class="mb-4">
                    <label class="form-label fw-semibold">Email Verification</label>
                    <?php if ($userDetails && isset($userDetails['email_verified']) && $userDetails['email_verified']): ?>
                        <div class="status-badge status-verified">
                            <i class="bi bi-check-circle-fill"></i>
                            <span>
                                Email Verified
                                <?php if ($userDetails['email_verified_at']): ?>
                                    on <?php echo date('M d, Y', strtotime($userDetails['email_verified_at'])); ?>
                                <?php endif; ?>
                            </span>
                        </div>
                    <?php else: ?>
                        <div class="status-badge status-pending">
                            <i class="bi bi-exclamation-circle-fill"></i>
                            <span>Email Not Verified</span>
                        </div>
                        <p class="text-muted small mt-2">Verify your email to unlock full features</p>
                        <button type="button" class="btn btn-warning btn-sm mt-2" data-bs-toggle="modal" data-bs-target="#verifyEmailModal">
                            <i class="bi bi-envelope-check"></i> Verify Email Now
                        </button>
                    <?php endif; ?>
                </div>

                <div class="d-flex gap-2 justify-content-between">
                    <a href="dashboard.php" class="btn btn-outline-secondary">
                        <i class="bi bi-arrow-left"></i> Back to Dashboard
                    </a>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-check-lg"></i> Save Changes
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>

<!-- Email Verification Modal -->
<div class="modal fade" id="verifyEmailModal" tabindex="-1" aria-labelledby="verifyEmailLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content" style="border-radius: 16px; border: none; box-shadow: 0 8px 32px rgba(0,0,0,0.15);">
            <div class="modal-header" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none;">
                <h5 class="modal-title" id="verifyEmailLabel">
                    <i class="bi bi-envelope-check"></i> Verify Email Address
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body p-4">
                <div id="verify-message"></div>
                
                <div class="mb-3">
                    <p class="text-muted mb-3">
                        We've sent a 6-digit OTP to <strong><?php echo htmlspecialchars($userDetails['email'] ?? ''); ?></strong>
                    </p>
                </div>

                <form id="otpForm" onsubmit="verifyOTP(event)">
                    <div class="mb-3">
                        <label class="form-label fw-semibold">Enter OTP Code</label>
                        <div style="display: flex; gap: 8px; justify-content: center;">
                            <input type="text" id="otp-input" class="form-control" placeholder="000000" maxlength="6" 
                                   style="font-size: 24px; text-align: center; letter-spacing: 8px; font-weight: bold;" required>
                        </div>
                        <small class="text-muted d-block mt-2 text-center">6-digit code from your email</small>
                    </div>

                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary btn-lg">Verify OTP</button>
                    </div>
                </form>

                <hr>

                <div class="text-center">
                    <small class="text-muted">Didn't receive the OTP?</small>
                    <br>
                    <button type="button" class="btn btn-link btn-sm mt-2" onclick="resendOTP()">
                        Resend OTP
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Verify OTP via AJAX
async function verifyOTP(e) {
    e.preventDefault();
    
    const otp = document.getElementById('otp-input').value.trim();
    const messageDiv = document.getElementById('verify-message');
    
    if (!otp || otp.length !== 6) {
        messageDiv.innerHTML = '<div class="alert alert-danger">Please enter a valid 6-digit OTP</div>';
        return;
    }
    
    try {
        const response = await fetch('../api/verify_otp_modal.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ 
                otp: otp,
                email: '<?php echo htmlspecialchars($userDetails['email']); ?>'
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            messageDiv.innerHTML = '<div class="alert alert-success"><i class="bi bi-check-circle"></i> ' + data.message + '</div>';
            document.getElementById('otpForm').style.display = 'none';
            
            // Reload page after 2 seconds
            setTimeout(() => {
                location.reload();
            }, 2000);
        } else {
            messageDiv.innerHTML = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> ' + data.message + '</div>';
        }
    } catch (error) {
        messageDiv.innerHTML = '<div class="alert alert-danger">Error: ' + error.message + '</div>';
    }
}

// Resend OTP
async function resendOTP() {
    try {
        const response = await fetch('../api/resend_otp_modal.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ 
                email: '<?php echo htmlspecialchars($userDetails['email']); ?>'
            })
        });
        
        const data = await response.json();
        const messageDiv = document.getElementById('verify-message');
        
        if (data.success) {
            messageDiv.innerHTML = '<div class="alert alert-success"><i class="bi bi-check-circle"></i> ' + data.message + '</div>';
            document.getElementById('otp-input').value = '';
            document.getElementById('otp-input').focus();
        } else {
            messageDiv.innerHTML = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> ' + data.message + '</div>';
        }
    } catch (error) {
        document.getElementById('verify-message').innerHTML = '<div class="alert alert-danger">Error: ' + error.message + '</div>';
    }
}
</script>
</body>
</html>
