<?php
// admin/approve_payment.php - Razorpay payment management
require_once(__DIR__ . '/../config/config.php');
require_once(__DIR__ . '/../classes/Database.php');
require_once(__DIR__ . '/../classes/Auth.php');
require_once(__DIR__ . '/../classes/RazorpayPayment.php');

session_start();

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn() || !$auth->isAdmin()) {
    http_response_code(403);
    die(json_encode(['status' => 'error', 'message' => 'Unauthorized']));
}

header('Content-Type: application/json');

$action = $_GET['action'] ?? $_POST['action'] ?? 'view';
$paymentId = $_GET['payment_id'] ?? $_POST['payment_id'] ?? null;
$subscriptionId = $_GET['subscription_id'] ?? $_POST['subscription_id'] ?? null;

if (!$paymentId && !$subscriptionId) {
    die(json_encode(['status' => 'error', 'message' => 'Payment/Subscription ID required']));
}

$razorpay = new RazorpayPayment(RAZORPAY_KEY_ID, RAZORPAY_KEY_SECRET);

switch ($action) {
    case 'view':
        // Get payment/subscription details
        if ($paymentId) {
            $paymentDetails = $razorpay->getPaymentDetails($paymentId);
            echo json_encode($paymentDetails);
        } else {
            $subscription = $db->fetch('SELECT * FROM user_subscriptions WHERE id = ?', [$subscriptionId]);
            if ($subscription) {
                echo json_encode(['status' => true, 'subscription' => $subscription]);
            } else {
                echo json_encode(['status' => false, 'message' => 'Subscription not found']);
            }
        }
        break;
    
    case 'refund':
        // Process refund for Razorpay payment
        $data = json_decode(file_get_contents('php://input'), true);
        $refundAmount = $data['amount'] ?? null;
        
        if (!$paymentId) {
            die(json_encode(['status' => false, 'message' => 'Payment ID required']));
        }
        
        $refundResult = $razorpay->refundPayment($paymentId, $refundAmount);
        
        if ($refundResult['status']) {
            // Log refund in subscription history
            $subscription = $db->fetch('SELECT * FROM user_subscriptions WHERE razorpay_payment_id = ?', [$paymentId]);
            if ($subscription) {
                $db->insert('subscription_history', [
                    'user_id' => $subscription['user_id'],
                    'subscription_id' => $subscription['id'],
                    'action' => 'refund_processed',
                    'old_status' => $subscription['status'],
                    'new_status' => 'cancelled',
                    'details' => json_encode(['refund_id' => $refundResult['refund_id'], 'amount' => $refundResult['amount']])
                ]);
                
                // Cancel subscription
                $db->update('user_subscriptions', ['status' => 'cancelled'], 'id', $subscription['id']);
            }
            echo json_encode(['status' => true, 'message' => 'Refund processed successfully', 'refund_id' => $refundResult['refund_id']]);
        } else {
            echo json_encode(['status' => false, 'message' => $refundResult['message']]);
        }
        break;
    
    default:
        echo json_encode(['status' => false, 'message' => 'Unknown action']);
}
?>
