<?php
/**
 * Admin Cache Management Dashboard
 * Search cache ko view aur manage karne ke liye
 */

session_start();

// Check admin access
if (!isset($_SESSION['admin_id'])) {
    die("❌ Admin access required!");
}

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/classes/Search.php';
require_once __DIR__ . '/classes/User.php';

$db = Database::getInstance();
$user = new User($db);
$search = new Search($db, $user);

$action = $_GET['action'] ?? 'view';
$type = $_GET['type'] ?? null;

// Handle actions
if ($_POST['action'] == 'clear_old') {
    $days = intval($_POST['days'] ?? 90);
    $result = $search->clearOldCache($days);
    echo json_encode($result);
    exit;
}

// Get all cached data
$cachedData = $search->getAllCachedData($type, 500);
?>

<!DOCTYPE html>
<html>
<head>
    <title>🔍 Search Cache Manager</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        
        h1 {
            color: #667eea;
            margin-top: 0;
        }
        
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        
        .stat-box {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
        }
        
        .stat-box h3 {
            margin: 0;
            font-size: 14px;
            opacity: 0.9;
        }
        
        .stat-box .number {
            font-size: 32px;
            font-weight: bold;
            margin: 10px 0 0 0;
        }
        
        .filters {
            margin: 20px 0;
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .filters a, .filters button {
            padding: 10px 15px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            background: #667eea;
            color: white;
            transition: all 0.3s;
        }
        
        .filters a:hover, .filters button:hover {
            background: #764ba2;
            transform: translateY(-2px);
        }
        
        .filters a.active {
            background: #764ba2;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        
        th {
            background: #f5f5f5;
            padding: 12px;
            text-align: left;
            border-bottom: 2px solid #667eea;
            font-weight: 600;
        }
        
        td {
            padding: 12px;
            border-bottom: 1px solid #eee;
        }
        
        tr:hover {
            background: #f9f9f9;
        }
        
        .badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .badge-mobile {
            background: #e3f2fd;
            color: #1976d2;
        }
        
        .badge-aadhaar {
            background: #f3e5f5;
            color: #7b1fa2;
        }
        
        .actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        
        .btn-danger {
            background: #f44336;
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            color: white;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .btn-danger:hover {
            background: #d32f2f;
            transform: translateY(-2px);
        }
        
        .alert {
            padding: 15px;
            border-radius: 5px;
            margin: 10px 0;
        }
        
        .alert-success {
            background: #c8e6c9;
            color: #2e7d32;
            border: 1px solid #81c784;
        }
        
        .alert-info {
            background: #bbdefb;
            color: #1565c0;
            border: 1px solid #64b5f6;
        }
        
        .empty {
            text-align: center;
            padding: 40px;
            color: #999;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Search Cache Manager</h1>
        
        <div class="alert alert-info">
            <strong>💡 Cache System:</strong> Jab API down ho, ye permanent cache use hoga. Har successful search automatically cache mein save hota hai.
        </div>
        
        <?php
            $mobileCount = count(array_filter($cachedData, fn($x) => $x['type'] == 'mobile'));
            $aadhaarCount = count(array_filter($cachedData, fn($x) => $x['type'] == 'aadhaar'));
            $totalQueries = array_sum(array_column($cachedData, 'data_count'));
        ?>
        
        <div class="stats">
            <div class="stat-box">
                <h3>Mobile Cache</h3>
                <div class="number"><?php echo $mobileCount; ?></div>
            </div>
            <div class="stat-box">
                <h3>Aadhaar Cache</h3>
                <div class="number"><?php echo $aadhaarCount; ?></div>
            </div>
            <div class="stat-box">
                <h3>Total Searches</h3>
                <div class="number"><?php echo $totalQueries; ?></div>
            </div>
            <div class="stat-box">
                <h3>Cache Size</h3>
                <div class="number"><?php echo count($cachedData); ?></div>
            </div>
        </div>
        
        <div class="filters">
            <a href="?type=" class="<?php echo !$type ? 'active' : ''; ?>">📊 All</a>
            <a href="?type=mobile" class="<?php echo $type === 'mobile' ? 'active' : ''; ?>">📱 Mobile Only</a>
            <a href="?type=aadhaar" class="<?php echo $type === 'aadhaar' ? 'active' : ''; ?>">🆔 Aadhaar Only</a>
        </div>
        
        <div class="actions">
            <button class="btn-danger" onclick="clearOldCache()">🗑️ Delete Cache older than 90 days</button>
        </div>
        
        <?php if (count($cachedData) > 0): ?>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Type</th>
                        <th>Query</th>
                        <th>Search Count</th>
                        <th>Last Updated</th>
                        <th>Created</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($cachedData as $item): ?>
                        <tr>
                            <td>#<?php echo $item['id']; ?></td>
                            <td>
                                <span class="badge badge-<?php echo $item['type']; ?>">
                                    <?php echo strtoupper($item['type']); ?>
                                </span>
                            </td>
                            <td><strong><?php echo htmlspecialchars($item['query']); ?></strong></td>
                            <td><strong><?php echo $item['data_count']; ?>x</strong></td>
                            <td><?php echo date('d M Y, h:i A', strtotime($item['last_updated'])); ?></td>
                            <td><?php echo date('d M Y', strtotime($item['created_at'])); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <div class="empty">
                <h2>📭 No cached data yet</h2>
                <p>Jaise hi users searches karengen, cache yahan dikhega.</p>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
        function clearOldCache() {
            if (confirm('⚠️ Are you sure? Ye 90+ days puraana data delete kar dega.')) {
                fetch('admin/cache_management.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: 'action=clear_old&days=90'
                })
                .then(r => r.json())
                .then(data => {
                    if (data.success) {
                        alert('✅ ' + data.message);
                        location.reload();
                    } else {
                        alert('❌ ' + data.message);
                    }
                });
            }
        }
    </script>
</body>
</html>
