<?php
// admin/manage_subscriptions.php - Admin subscription management page
session_start();
require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';
require_once '../classes/SubscriptionManager.php';

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn() || !$auth->isAdmin()) {
    header('Location: ../login.php');
    exit();
}

$subscriptionMgr = new SubscriptionManager();
$message = '';
$message_type = '';

// Handle manual subscription addition
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'add_subscription') {
        $user_id = (int)($_POST['user_id'] ?? 0);
        $plan_id = (int)($_POST['plan_id'] ?? 0);
        
        if (!$user_id || !$plan_id) {
            $message = 'Please select both user and plan';
            $message_type = 'danger';
        } else {
            $user = $db->fetch('SELECT id FROM users WHERE id = ?', [$user_id]);
            if (!$user) {
                $message = 'User not found';
                $message_type = 'danger';
            } else {
                $plan = $db->fetch('SELECT * FROM subscription_plans WHERE id = ?', [$plan_id]);
                if (!$plan) {
                    $message = 'Plan not found';
                    $message_type = 'danger';
                } else {
                    // Add subscription
                    $start_date = date('Y-m-d H:i:s');
                    $end_date = date('Y-m-d H:i:s', strtotime("+{$plan['duration_days']} days"));
                    
                    error_log("ADMIN: Adding subscription - User: $user_id, Plan: $plan_id, End: $end_date");
                    
                    $result = $db->insert('user_subscriptions', [
                        'user_id' => $user_id,
                        'plan_id' => $plan_id,
                        'status' => 'active',
                        'payment_status' => 'success',
                        'start_date' => $start_date,
                        'end_date' => $end_date,
                        'search_used' => 0,
                        'created_at' => $start_date
                    ]);
                    
                    if ($result) {
                        error_log("✓ Subscription added successfully. ID: $result");
                    } else {
                        error_log("✗ Failed to add subscription to database");
                    }
                    
                    // Log to history
                    $db->insert('subscription_history', [
                        'user_id' => $user_id,
                        'action' => 'manual_admin_add',
                        'old_status' => 'none',
                        'new_status' => 'active',
                        'details' => json_encode([
                            'plan_id' => $plan_id,
                            'plan_name' => $plan['plan_name'],
                            'added_by_admin' => true
                        ])
                    ]);
                    
                    $message = '✅ Subscription added successfully!';
                    $message_type = 'success';
                }
            }
        }
    }
}

// Get all users and plans
$users = $db->fetchAll('SELECT id, name, email FROM users WHERE role != "admin" ORDER BY name');
$plans = $db->fetchAll('SELECT * FROM subscription_plans WHERE is_active = 1 ORDER BY price');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Subscriptions - Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        :root {
            --primary: #007aff;
            --secondary: #f2f2f7;
            --text: #1c1c1e;
            --shadow: rgba(0,0,0,0.1);
        }

        body {
            background: var(--secondary);
            color: var(--text);
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }

        .sidebar {
            height: 100vh;
            background: white;
            padding: 1.5rem;
            box-shadow: 2px 0 10px var(--shadow);
            position: fixed;
            left: 0;
            top: 0;
            width: 240px;
            z-index: 2000;
            overflow-y: auto;
            display: flex;
            flex-direction: column;
        }

        .sidebar h5 {
            color: var(--primary);
            text-align: center;
            margin-bottom: 1.5rem;
            font-weight: 700;
        }

        .sidebar a {
            display: flex;
            align-items: center;
            color: var(--text);
            text-decoration: none;
            padding: 0.75rem 1rem;
            margin-bottom: 0.5rem;
            border-radius: 10px;
            font-size: 15px;
            transition: 0.2s;
        }

        .sidebar a.active,
        .sidebar a:hover {
            background: var(--primary);
            color: #fff !important;
        }

        .sidebar i {
            margin-right: 0.6rem;
        }

        .main-content {
            padding: 2rem;
            margin-left: 240px;
        }

        .card {
            border: none;
            border-radius: 16px;
            box-shadow: 0 4px 18px var(--shadow);
        }

        .btn-primary {
            background: linear-gradient(135deg, #007AFF 0%, #0051D5 100%);
            border: none;
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, #0051D5 0%, #003DA0 100%);
        }

        .form-section {
            background: white;
            padding: 30px;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            margin-bottom: 30px;
        }

        .alert {
            border-radius: 12px;
            border: none;
        }

        .form-control, .form-select {
            border: 1px solid #e5e5ea;
            border-radius: 8px;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--primary);
            box-shadow: 0 0 0 0.2rem rgba(0, 122, 255, 0.25);
        }

        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
            }
            .main-content {
                margin-left: 0;
                padding: 1rem;
            }
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 sidebar">
                <h5>📊 Admin Panel</h5>
                <a href="dashboard.php">
                    <i data-feather="home"></i> Dashboard
                </a>
                <a href="subscription_plans.php">
                    <i data-feather="package"></i> Manage Plans
                </a>
                <a href="manage_subscriptions.php" class="active">
                    <i data-feather="user-plus"></i> Add to User
                </a>
                <a href="payment_management.php">
                    <i data-feather="trending-up"></i> Payment Reports
                </a>
                <a href="dashboard.php?page=users">
                    <i data-feather="users"></i> Users
                </a>
                <hr>
                <a href="../logout.php" style="color: #FF3B30;">
                    <i data-feather="log-out"></i> Logout
                </a>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 main-content">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>💳 Manage User Subscriptions</h2>
                </div>

                <!-- Messages -->
                <?php if ($message): ?>
                    <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show">
                        <?php echo $message; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Add Subscription Form -->
                <div class="form-section">
                    <h4 class="mb-4">➕ Add Subscription to User</h4>
                    <form method="POST">
                        <input type="hidden" name="action" value="add_subscription">

                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Select User</label>
                                <select name="user_id" class="form-select" required>
                                    <option value="">-- Choose User --</option>
                                    <?php foreach ($users as $user): ?>
                                        <option value="<?php echo $user['id']; ?>">
                                            <?php echo htmlspecialchars($user['name']) . ' (' . htmlspecialchars($user['email']) . ')'; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="col-md-4 mb-3">
                                <label class="form-label">Select Plan</label>
                                <select name="plan_id" class="form-select" required>
                                    <option value="">-- Choose Plan --</option>
                                    <?php foreach ($plans as $plan): ?>
                                        <option value="<?php echo $plan['id']; ?>">
                                            <?php echo htmlspecialchars($plan['plan_name']) . ' - ₹' . number_format($plan['price'], 0); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="col-md-4 mb-3" style="display: flex; align-items: flex-end;">
                                <button type="submit" class="btn btn-primary w-100">
                                    <i data-feather="plus" style="width: 18px; height: 18px;"></i> Add Subscription
                                </button>
                            </div>
                        </div>
                    </form>
                </div>

                <!-- Plans Overview -->
                <div class="form-section">
                    <h4 class="mb-4">📦 Available Plans</h4>
                    <div class="row">
                        <?php foreach ($plans as $plan): ?>
                            <div class="col-md-6 col-lg-4 mb-3">
                                <div class="card" style="height: 100%;">
                                    <div class="card-body">
                                        <h5 class="card-title"><?php echo htmlspecialchars($plan['plan_name']); ?></h5>
                                        <div style="margin: 15px 0;">
                                            <div class="mb-2">
                                                <small class="text-muted">Price</small>
                                                <p style="font-size: 24px; font-weight: bold; color: #34C759; margin: 0;">₹<?php echo number_format($plan['price'], 0); ?></p>
                                            </div>
                                            <div class="mb-2">
                                                <small class="text-muted">Duration</small>
                                                <p style="margin: 0;"><?php echo $plan['duration_days']; ?> days</p>
                                            </div>
                                            <div class="mb-2">
                                                <small class="text-muted">Search Limit</small>
                                                <p style="margin: 0;"><?php echo $plan['search_limit']; ?> searches</p>
                                            </div>
                                        </div>
                                        <?php if (!empty($plan['description'])): ?>
                                            <small class="text-muted"><?php echo htmlspecialchars($plan['description']); ?></small>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>feather.replace();</script>
</body>
</html>
