<?php
/**
 * Device Restriction Settings
 * Admin can set restrictions on which devices can access the app
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../classes/Database.php';
require_once __DIR__ . '/../classes/Auth.php';

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn() || !$auth->isAdmin()) {
    header('Location: ../login.php');
    exit();
}

// Handle save settings
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_restrictions'])) {
    $block_mobile = isset($_POST['block_mobile']) ? 1 : 0;
    $block_tablet = isset($_POST['block_tablet']) ? 1 : 0;
    $block_desktop = isset($_POST['block_desktop']) ? 1 : 0;
    $restriction_message = trim($_POST['restriction_message'] ?? '');
    
    // Save to database
    $keys = [
        'device_block_mobile' => $block_mobile,
        'device_block_tablet' => $block_tablet,
        'device_block_desktop' => $block_desktop,
        'device_restriction_message' => $restriction_message
    ];
    
    foreach ($keys as $key => $value) {
        $exists = $db->fetch('SELECT id FROM admin_settings WHERE setting_key = ?', [$key]);
        
        if ($exists) {
            $db->update('admin_settings', 
                ['setting_value' => $value],
                'setting_key = ?',
                [$key]
            );
        } else {
            $db->insert('admin_settings',
                ['setting_key' => $key, 'setting_value' => $value]
            );
        }
    }
    
    echo '<div class="alert alert-success alert-dismissible fade show" role="alert">
            ✓ Device restrictions updated successfully!
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>';
}

// Fetch current settings
$block_mobile = $db->fetch('SELECT setting_value FROM admin_settings WHERE setting_key = ?', ['device_block_mobile']);
$block_tablet = $db->fetch('SELECT setting_value FROM admin_settings WHERE setting_key = ?', ['device_block_tablet']);
$block_desktop = $db->fetch('SELECT setting_value FROM admin_settings WHERE setting_key = ?', ['device_block_desktop']);
$restriction_message = $db->fetch('SELECT setting_value FROM admin_settings WHERE setting_key = ?', ['device_restriction_message']);

$block_mobile_val = $block_mobile['setting_value'] ?? 0;
$block_tablet_val = $block_tablet['setting_value'] ?? 0;
$block_desktop_val = $block_desktop['setting_value'] ?? 0;
$message = $restriction_message['setting_value'] ?? 'Your device is not allowed to access this application.';

?>

<div class="container-fluid">
    <div style="margin-bottom: 20px;">
        <h4 class="mb-0">🔒 Device Restrictions</h4>
        <small class="text-muted">Control which devices can access the application</small>
    </div>

    <div class="card shadow-sm border-0 mb-4">
        <div class="card-header" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none;">
            <h5 class="mb-0">Device Access Control</h5>
        </div>
        
        <div class="card-body">
            <form method="POST">
                <input type="hidden" name="save_restrictions" value="1">
                
                <div class="row">
                    <div class="col-md-4">
                        <div class="form-check form-switch mb-3">
                            <input class="form-check-input" type="checkbox" name="block_mobile" id="blockMobile" 
                                   <?= $block_mobile_val ? 'checked' : '' ?> style="width: 50px; height: 30px;">
                            <label class="form-check-label" for="blockMobile">
                                <strong>📱 Block Mobile Devices</strong>
                                <small class="d-block text-muted">Prevent access from mobile phones</small>
                            </label>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-check form-switch mb-3">
                            <input class="form-check-input" type="checkbox" name="block_tablet" id="blockTablet" 
                                   <?= $block_tablet_val ? 'checked' : '' ?> style="width: 50px; height: 30px;">
                            <label class="form-check-label" for="blockTablet">
                                <strong>📱 Block Tablets</strong>
                                <small class="d-block text-muted">Prevent access from tablets</small>
                            </label>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-check form-switch mb-3">
                            <input class="form-check-input" type="checkbox" name="block_desktop" id="blockDesktop" 
                                   <?= $block_desktop_val ? 'checked' : '' ?> style="width: 50px; height: 30px;">
                            <label class="form-check-label" for="blockDesktop">
                                <strong>💻 Block Desktop</strong>
                                <small class="d-block text-muted">Prevent access from desktop computers</small>
                            </label>
                        </div>
                    </div>
                </div>
                
                <hr>
                
                <div class="mb-3">
                    <label class="form-label">Restriction Message</label>
                    <textarea name="restriction_message" class="form-control" rows="3" placeholder="Message to show when user is blocked"><?= htmlspecialchars($message) ?></textarea>
                    <small class="text-muted">This message will be displayed to users accessing from blocked devices</small>
                </div>
                
                <button type="submit" class="btn btn-primary btn-lg">Save Restrictions</button>
            </form>
        </div>
    </div>

    <div class="card shadow-sm border-0">
        <div class="card-header" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none;">
            <h5 class="mb-0">Current Settings Summary</h5>
        </div>
        
        <div class="card-body">
            <div class="row">
                <div class="col-md-6">
                    <h6>Blocked Devices:</h6>
                    <ul class="list-unstyled">
                        <li><?= $block_mobile_val ? '❌ Mobile Devices - BLOCKED' : '✅ Mobile Devices - ALLOWED' ?></li>
                        <li><?= $block_tablet_val ? '❌ Tablets - BLOCKED' : '✅ Tablets - ALLOWED' ?></li>
                        <li><?= $block_desktop_val ? '❌ Desktop - BLOCKED' : '✅ Desktop - ALLOWED' ?></li>
                    </ul>
                </div>
                
                <div class="col-md-6">
                    <h6>Restriction Message:</h6>
                    <p class="alert alert-info mb-0"><?= htmlspecialchars($message) ?></p>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.form-check-input {
    appearance: none;
    width: 50px;
    height: 30px;
    background-color: #ccc;
    border-radius: 20px;
    cursor: pointer;
    border: 2px solid #ccc;
    transition: all 0.3s ease;
    position: relative;
}

.form-check-input:checked {
    background-color: #dc3545;
    border-color: #dc3545;
}

.form-check-input::after {
    content: '';
    position: absolute;
    width: 24px;
    height: 24px;
    background-color: white;
    border-radius: 50%;
    top: 2px;
    left: 2px;
    transition: all 0.3s ease;
}

.form-check-input:checked::after {
    left: 22px;
}

.form-check-label {
    margin-left: 10px;
    cursor: pointer;
}
</style>
