<?php
session_start();
require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn() || !$auth->isAdmin()) {
    header('Location: ../login.php');
    exit();
}

// Get payment statistics
$totalResult = $db->fetch('SELECT COUNT(*) as count FROM user_subscriptions WHERE payment_status = "success"');
$revenueResult = $db->fetch('SELECT SUM(sp.price) as total FROM user_subscriptions us JOIN subscription_plans sp ON us.plan_id = sp.id WHERE us.payment_status = "success"');
$pendingResult = $db->fetch('SELECT COUNT(*) as count FROM user_subscriptions WHERE payment_status = "pending"');
$failedResult = $db->fetch('SELECT COUNT(*) as count FROM user_subscriptions WHERE payment_status = "failed"');

$stats = [
    'total_payments' => is_array($totalResult) ? $totalResult['count'] : 0,
    'total_revenue' => is_array($revenueResult) ? ($revenueResult['total'] ?? 0) : 0,
    'pending_payments' => is_array($pendingResult) ? $pendingResult['count'] : 0,
    'failed_payments' => is_array($failedResult) ? $failedResult['count'] : 0,
];

// Get all subscriptions with payment details
$subscriptions = $db->fetchAll(
    "SELECT us.*, sp.plan_name, sp.price, u.name, u.email 
     FROM user_subscriptions us 
     JOIN subscription_plans sp ON us.plan_id = sp.id 
     JOIN users u ON us.user_id = u.id 
     ORDER BY us.created_at DESC"
) ?? [];

// Handle AJAX requests for refund
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    if ($_POST['action'] === 'refund') {
        $subscriptionId = (int)($_POST['subscription_id'] ?? 0);
        $subscription = $db->fetch('SELECT * FROM user_subscriptions WHERE id = ?', [$subscriptionId]);
        
        if (!$subscription) {
            echo json_encode(['success' => false, 'message' => 'Subscription not found']);
            exit;
        }
        
        // Update subscription status to cancelled
        $db->update('user_subscriptions', 
            ['status' => 'cancelled', 'payment_status' => 'refunded'],
            'id = ?',
            [$subscriptionId]
        );
        
        echo json_encode(['success' => true, 'message' => 'Refund processed successfully']);
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Reports - Admin Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        :root {
            --primary: #007aff;
            --secondary: #f2f2f7;
            --text: #1c1c1e;
            --shadow: rgba(0,0,0,0.1);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            background: var(--secondary);
            color: var(--text);
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }

        .wrapper {
            display: flex;
            min-height: 100vh;
        }

        /* SIDEBAR */
        .sidebar {
            height: 100vh;
            background: white;
            padding: 1.5rem;
            box-shadow: 2px 0 10px var(--shadow);
            position: fixed;
            left: 0;
            top: 0;
            width: 240px;
            z-index: 2000;
            overflow-y: scroll !important;
            overflow-x: hidden;
            display: flex;
            flex-direction: column;
            scrollbar-width: auto;
        }

        .sidebar::-webkit-scrollbar {
            width: 10px;
        }

        .sidebar::-webkit-scrollbar-track {
            background: #f5f5f5;
            border-radius: 10px;
        }

        .sidebar::-webkit-scrollbar-thumb {
            background: #bbb;
            border-radius: 5px;
            border: 2px solid #f5f5f5;
        }

        .sidebar::-webkit-scrollbar-thumb:hover {
            background: #888;
        }

        .sidebar h5 {
            color: var(--primary);
            text-align: center;
            margin-bottom: 1.5rem;
            font-weight: 700;
        }

        .sidebar a {
            display: flex;
            align-items: center;
            color: var(--text);
            text-decoration: none;
            padding: 0.75rem 1rem;
            margin-bottom: 0.5rem;
            border-radius: 10px;
            font-size: 15px;
            transition: 0.2s;
        }

        .sidebar a.active,
        .sidebar a:hover {
            background: var(--primary);
            color: #fff !important;
        }

        .sidebar i {
            margin-right: 0.6rem;
        }

        .sidebar hr {
            margin: 1rem 0;
            opacity: 0.3;
        }

        /* MAIN CONTENT */
        .main-content {
            padding: 2rem;
            margin-left: 240px;
            flex: 1;
            width: calc(100% - 240px);
        }

        .card {
            border: none;
            border-radius: 16px;
            box-shadow: 0 4px 18px var(--shadow);
        }

        .stat-card {
            padding: 1.5rem;
            border-left: 4px solid var(--primary);
        }

        .stat-card h6 {
            color: #888;
            font-size: 0.85rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 0.5rem;
            font-weight: 600;
        }

        .stat-card h2 {
            color: var(--primary);
            font-weight: 700;
            margin: 0;
            font-size: 28px;
        }

        .page-title {
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid var(--secondary);
        }

        .page-title h2 {
            font-weight: 700;
            font-size: 28px;
            margin: 0;
        }

        .table th {
            background: var(--secondary);
            border: none;
            font-weight: 600;
            padding: 1rem;
        }

        .table td {
            padding: 1rem;
            vertical-align: middle;
        }

        .badge {
            padding: 0.5rem 0.75rem;
            border-radius: 6px;
            font-size: 0.85rem;
            font-weight: 600;
        }

        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
            }
            .main-content {
                margin-left: 0;
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="wrapper">
        <!-- Sidebar -->
        <div class="sidebar">
            <h5>📊 Admin Panel</h5>
            
            <a href="dashboard.php">
                <i data-feather="home"></i> Dashboard
            </a>
            <a href="subscription_plans.php">
                <i data-feather="package"></i> Manage Plans
            </a>
            <a href="manage_subscriptions.php">
                <i data-feather="user-plus"></i> Add to User
            </a>
            <a href="payment_management.php" class="active">
                <i data-feather="trending-up"></i> Payment Reports
            </a>
            <a href="dashboard.php?page=users">
                <i data-feather="users"></i> Users
            </a>
            
            <hr>
            
            <a href="../logout.php" style="color: #FF3B30;">
                <i data-feather="log-out"></i> Logout
            </a>
        </div>

        <!-- Main Content -->
        <div class="main-content">
            <div class="page-title">
                <h2>💳 Payment Reports</h2>
            </div>

            <!-- Stats -->
            <div class="row mb-4">
                <div class="col-md-3">
                    <div class="card stat-card">
                        <h6>Total Revenue</h6>
                        <h2>₹<?php echo number_format($stats['total_revenue'], 0); ?></h2>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card stat-card" style="border-left-color: #34C759;">
                        <h6>Successful Payments</h6>
                        <h2 style="color: #34C759;"><?php echo $stats['total_payments']; ?></h2>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card stat-card" style="border-left-color: #FF9500;">
                        <h6>Pending Payments</h6>
                        <h2 style="color: #FF9500;"><?php echo $stats['pending_payments']; ?></h2>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card stat-card" style="border-left-color: #FF3B30;">
                        <h6>Failed Payments</h6>
                        <h2 style="color: #FF3B30;"><?php echo $stats['failed_payments']; ?></h2>
                    </div>
                </div>
            </div>

            <!-- Transactions Table -->
            <div class="card">
                <div class="card-body">
                    <h5 class="mb-3">All Transactions</h5>

                    <?php if (empty($subscriptions)): ?>
                        <div class="alert alert-info">
                            <strong>ℹ️ No transactions found</strong>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>User Name</th>
                                        <th>Plan</th>
                                        <th>Amount</th>
                                        <th>Payment Status</th>
                                        <th>Date</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($subscriptions as $sub): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($sub['name']); ?></strong></td>
                                        <td><?php echo htmlspecialchars($sub['plan_name']); ?></td>
                                        <td><strong>₹<?php echo number_format($sub['price'], 2); ?></strong></td>
                                        <td>
                                            <?php 
                                            $statusColor = 'secondary';
                                            if ($sub['payment_status'] === 'success') $statusColor = 'success';
                                            elseif ($sub['payment_status'] === 'pending') $statusColor = 'warning';
                                            elseif ($sub['payment_status'] === 'failed') $statusColor = 'danger';
                                            ?>
                                            <span class="badge bg-<?php echo $statusColor; ?>">
                                                <?php echo ucfirst($sub['payment_status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo date('M d, Y H:i', strtotime($sub['created_at'])); ?></td>
                                        <td>
                                            <?php if ($sub['payment_status'] === 'success' && $sub['status'] === 'active'): ?>
                                                <button class="btn btn-sm btn-danger" onclick="refundPayment(<?php echo $sub['id']; ?>, '<?php echo htmlspecialchars($sub['name']); ?>')">
                                                    Refund
                                                </button>
                                            <?php else: ?>
                                                <span style="color: #999;">-</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        feather.replace();

        function refundPayment(subscriptionId, userName) {
            if (!confirm('Process refund for ' + userName + '?')) return;

            fetch('payment_management.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded'
                },
                body: 'action=refund&subscription_id=' + subscriptionId
            })
            .then(response => response.json())
            .then(data => {
                alert(data.message);
                if (data.success) location.reload();
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred');
            });
        }
    </script>
</body>
</html>
