<?php
// admin/reject_payment.php - Reject/Cancel Razorpay subscriptions
require_once(__DIR__ . '/../config/config.php');
require_once(__DIR__ . '/../classes/Database.php');
require_once(__DIR__ . '/../classes/Auth.php');
require_once(__DIR__ . '/../classes/RazorpayPayment.php');

session_start();

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn() || !$auth->isAdmin()) {
    http_response_code(403);
    die(json_encode(['status' => 'error', 'message' => 'Unauthorized']));
}

header('Content-Type: application/json');

$data = json_decode(file_get_contents('php://input'), true);
$subscriptionId = $data['subscription_id'] ?? null;
$reason = $data['reason'] ?? 'Admin rejection';
$refund = $data['refund'] ?? false;

if (!$subscriptionId) {
    die(json_encode(['status' => 'error', 'message' => 'Subscription ID required']));
}

$subscription = $db->fetch('SELECT * FROM user_subscriptions WHERE id = ?', [$subscriptionId]);

if (!$subscription) {
    die(json_encode(['status' => 'error', 'message' => 'Subscription not found']));
}

try {
    $refundId = null;
    
    // If refund is requested and payment was successful
    if ($refund && $subscription['razorpay_payment_id'] && $subscription['payment_status'] === 'success') {
        $razorpay = new RazorpayPayment(RAZORPAY_KEY_ID, RAZORPAY_KEY_SECRET);
        $refundResult = $razorpay->refundPayment($subscription['razorpay_payment_id']);
        
        if (!$refundResult['status']) {
            die(json_encode(['status' => false, 'message' => 'Failed to process refund: ' . $refundResult['message']]));
        }
        
        $refundId = $refundResult['refund_id'];
    }
    
    // Cancel subscription
    $db->update('user_subscriptions', 
        ['status' => 'cancelled'], 
        'id', 
        $subscriptionId
    );
    
    // Log in subscription history
    $db->insert('subscription_history', [
        'user_id' => $subscription['user_id'],
        'subscription_id' => $subscriptionId,
        'action' => 'cancelled_by_admin',
        'old_status' => $subscription['status'],
        'new_status' => 'cancelled',
        'details' => json_encode([
            'reason' => $reason,
            'refund_processed' => $refund ? 'yes' : 'no',
            'refund_id' => $refundId
        ])
    ]);
    
    echo json_encode([
        'status' => true, 
        'message' => 'Subscription cancelled' . ($refund ? ' with refund' : ''),
        'refund_id' => $refundId
    ]);
    
} catch (Exception $e) {
    error_log('Reject Payment Error: ' . $e->getMessage());
    echo json_encode(['status' => false, 'message' => 'Error: ' . $e->getMessage()]);
}
?>
