<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 0);

require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn() || !$auth->isAdmin()) {
    header('Location: ../login.php');
    exit();
}

// Auto-create subscription_plans table if it doesn't exist
try {
    $tableCheck = $db->fetch("SELECT TABLE_NAME FROM information_schema.TABLES WHERE TABLE_SCHEMA = 'iplztpse_final' AND TABLE_NAME = 'subscription_plans'");
    
    if (!$tableCheck) {
        $db->query("CREATE TABLE subscription_plans (
            id INT PRIMARY KEY AUTO_INCREMENT,
            plan_name VARCHAR(255) NOT NULL,
            duration_days INT NOT NULL,
            search_limit INT NOT NULL,
            price DECIMAL(10, 2) NOT NULL,
            description TEXT,
            card_color VARCHAR(7) DEFAULT '#007AFF',
            is_active TINYINT DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )");
    } else {
        // Check if card_color column exists, if not add it
        $columnCheck = $db->fetch("SELECT COLUMN_NAME FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = 'iplztpse_final' AND TABLE_NAME = 'subscription_plans' AND COLUMN_NAME = 'card_color'");
        
        if (!$columnCheck) {
            $db->query("ALTER TABLE subscription_plans ADD COLUMN card_color VARCHAR(7) DEFAULT '#007AFF' AFTER description");
        }
    }
} catch (Exception $e) {
    error_log('Table creation error: ' . $e->getMessage());
}

$message = '';
$message_type = '';

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'create_plan') {
        $planName = trim($_POST['plan_name'] ?? '');
        $durationDays = (int)($_POST['duration_days'] ?? 0);
        $searchLimit = (int)($_POST['search_limit'] ?? 0);
        $price = (float)($_POST['price'] ?? 0);
        $description = trim($_POST['description'] ?? '');
        $cardColor = trim($_POST['card_color'] ?? '#007AFF');
        
        if (!$planName || $durationDays <= 0 || $searchLimit <= 0 || $price <= 0) {
            $message = '❌ All fields are required and must be valid';
            $message_type = 'danger';
        } else {
            try {
                $insertId = $db->insert('subscription_plans', [
                    'plan_name' => $planName,
                    'duration_days' => $durationDays,
                    'search_limit' => $searchLimit,
                    'price' => $price,
                    'description' => $description,
                    'card_color' => $cardColor,
                    'is_active' => 1,
                    'created_at' => date('Y-m-d H:i:s')
                ]);
                
                if ($insertId) {
                    $message = '✅ Plan created successfully!';
                    $message_type = 'success';
                } else {
                    $message = '❌ Failed to create plan - database error';
                    $message_type = 'danger';
                }
            } catch (Exception $e) {
                $message = '❌ Error: ' . $e->getMessage();
                $message_type = 'danger';
                error_log('Plan Creation Error: ' . $e->getMessage());
            }
        }
    } elseif ($action === 'update_plan') {
        $planId = (int)($_POST['plan_id'] ?? 0);
        $planName = trim($_POST['plan_name'] ?? '');
        $durationDays = (int)($_POST['duration_days'] ?? 0);
        $searchLimit = (int)($_POST['search_limit'] ?? 0);
        $price = (float)($_POST['price'] ?? 0);
        $description = trim($_POST['description'] ?? '');
        $cardColor = trim($_POST['card_color'] ?? '#007AFF');
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        
        if (!$planId || !$planName || $durationDays <= 0 || $searchLimit <= 0 || $price <= 0) {
            $message = '❌ All fields are required and must be valid';
            $message_type = 'danger';
        } else {
            try {
                $db->query(
                    'UPDATE subscription_plans SET plan_name = ?, duration_days = ?, search_limit = ?, price = ?, description = ?, card_color = ?, is_active = ? WHERE id = ?',
                    [$planName, $durationDays, $searchLimit, $price, $description, $cardColor, $isActive, $planId]
                );
                $message = '✅ Plan updated successfully!';
                $message_type = 'success';
            } catch (Exception $e) {
                $message = '❌ Error: ' . $e->getMessage();
                $message_type = 'danger';
            }
        }
    } elseif ($action === 'delete_plan') {
        $planId = (int)($_POST['plan_id'] ?? 0);
        if ($planId > 0) {
            try {
                $db->query('DELETE FROM subscription_plans WHERE id = ?', [$planId]);
                $message = '✅ Plan deleted successfully!';
                $message_type = 'success';
            } catch (Exception $e) {
                $message = '❌ Error: ' . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }
}

// Load plans - after POST so new plans are visible
$plans = $db->fetchAll('SELECT * FROM subscription_plans WHERE is_active = 1 ORDER BY duration_days ASC') ?? [];

// Get stats
$totalPlans = count($plans);
$activeSubscriptions = $db->fetch("SELECT COUNT(*) as count FROM user_subscriptions WHERE status = 'active'")['count'] ?? 0;
$usersOnTrial = $db->fetch("SELECT COUNT(*) as count FROM free_trial_tracking WHERE status = 'active'")['count'] ?? 0;
$revenue = $db->fetch("SELECT COALESCE(SUM(amount), 0) as total FROM payments WHERE status = 'completed'")['total'] ?? 0;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Subscription Plans - Admin Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        :root {
            --primary: #007aff;
            --secondary: #f2f2f7;
            --text: #1c1c1e;
            --shadow: rgba(0,0,0,0.1);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            background: var(--secondary);
            color: var(--text);
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
        }

        .wrapper {
            display: flex;
            min-height: 100vh;
        }

        /* SIDEBAR */
        .sidebar {
            height: 100vh;
            background: white;
            padding: 1.5rem;
            box-shadow: 2px 0 10px var(--shadow);
            position: fixed;
            left: 0;
            top: 0;
            width: 240px;
            z-index: 2000;
            overflow-y: scroll !important;
            overflow-x: hidden;
            display: flex;
            flex-direction: column;
            scrollbar-width: auto;
        }

        .sidebar::-webkit-scrollbar {
            width: 10px;
        }

        .sidebar::-webkit-scrollbar-track {
            background: #f5f5f5;
            border-radius: 10px;
        }

        .sidebar::-webkit-scrollbar-thumb {
            background: #bbb;
            border-radius: 5px;
            border: 2px solid #f5f5f5;
        }

        .sidebar::-webkit-scrollbar-thumb:hover {
            background: #888;
        }

        .sidebar h5 {
            color: var(--primary);
            text-align: center;
            margin-bottom: 1.5rem;
            font-weight: 700;
            font-size: 18px;
        }

        .sidebar a {
            display: flex;
            align-items: center;
            color: var(--text);
            text-decoration: none;
            padding: 0.75rem 1rem;
            margin-bottom: 0.5rem;
            border-radius: 10px;
            font-size: 15px;
            transition: 0.2s all;
            cursor: pointer;
        }

        .sidebar a:hover {
            background: var(--secondary);
        }

        .sidebar a.active {
            background: var(--primary);
            color: #fff;
        }

        .sidebar i {
            margin-right: 0.8rem;
            width: 20px;
            height: 20px;
        }

        .sidebar hr {
            margin: 1rem 0;
            opacity: 0.3;
        }

        /* MAIN CONTENT */
        .main-content {
            margin-left: 240px;
            flex: 1;
            padding: 2rem;
            width: calc(100% - 240px);
        }

        .card {
            border: none;
            border-radius: 16px;
            box-shadow: 0 4px 18px var(--shadow);
            margin-bottom: 2rem;
        }

        .stat-card {
            padding: 1.5rem;
            border-left: 4px solid var(--primary);
        }

        .stat-card h6 {
            color: #888;
            font-size: 0.85rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 0.5rem;
            font-weight: 600;
        }

        .stat-card h2 {
            color: var(--primary);
            font-weight: 700;
            margin: 0;
            font-size: 28px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #007AFF 0%, #0051D5 100%);
            border: none;
            font-weight: 600;
            padding: 0.6rem 1.2rem;
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, #0051D5 0%, #003DA0 100%);
        }

        .alert {
            border-radius: 12px;
            border: none;
            padding: 1rem 1.5rem;
        }

        .table th {
            background: var(--secondary);
            border: none;
            font-weight: 600;
            padding: 1rem;
        }

        .table td {
            padding: 1rem;
            vertical-align: middle;
        }

        .modal-content {
            border-radius: 12px;
            border: none;
        }

        .modal-header {
            border-bottom: 1px solid var(--secondary);
            padding: 1.5rem;
        }

        .form-control, .form-select {
            border: 1px solid #e5e5ea;
            border-radius: 8px;
            padding: 0.6rem 0.9rem;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--primary);
            box-shadow: 0 0 0 0.2rem rgba(0, 122, 255, 0.25);
        }

        .form-label {
            font-weight: 600;
            margin-bottom: 0.5rem;
        }

        .page-title {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid var(--secondary);
        }

        .page-title h2 {
            font-weight: 700;
            font-size: 28px;
            margin: 0;
        }

        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
                padding: 1rem;
            }

            .main-content {
                margin-left: 0;
                width: 100%;
                padding: 1rem;
            }

            .page-title {
                flex-direction: column;
                gap: 1rem;
            }
        }
    </style>
</head>
<body>
    <div class="wrapper">
        <!-- Sidebar -->
        <div class="sidebar">
            <h5>📊 Admin Panel</h5>
            
            <a href="dashboard.php">
                <i data-feather="home"></i> Dashboard
            </a>
            <a href="subscription_plans.php" class="active">
                <i data-feather="package"></i> Manage Plans
            </a>
            <a href="manage_subscriptions.php">
                <i data-feather="user-plus"></i> Add to User
            </a>
            <a href="payment_management.php">
                <i data-feather="trending-up"></i> Payment Reports
            </a>
            <a href="dashboard.php?page=users">
                <i data-feather="users"></i> Users
            </a>
            
            <hr>
            
            <a href="../logout.php" style="color: #FF3B30;">
                <i data-feather="log-out"></i> Logout
            </a>
        </div>

        <!-- Main Content -->
        <div class="main-content">
            <div class="page-title">
                <h2>💳 Subscription Plans</h2>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createPlanModal">
                    <i data-feather="plus" style="width: 18px; height: 18px; margin-right: 0.5rem;"></i>New Plan
                </button>
            </div>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show">
                    <?php echo $message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <!-- Stats -->
            <div class="row mb-4">
                <div class="col-md-3">
                    <div class="card stat-card">
                        <h6>Total Plans</h6>
                        <h2><?php echo $totalPlans; ?></h2>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card stat-card" style="border-left-color: #34C759;">
                        <h6>Active Subscriptions</h6>
                        <h2 style="color: #34C759;"><?php echo $activeSubscriptions; ?></h2>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card stat-card" style="border-left-color: #FF9500;">
                        <h6>Users on Trial</h6>
                        <h2 style="color: #FF9500;"><?php echo $usersOnTrial; ?></h2>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card stat-card" style="border-left-color: #34C759;">
                        <h6>Total Revenue</h6>
                        <h2 style="color: #34C759;">₹<?php echo number_format($revenue, 2); ?></h2>
                    </div>
                </div>
            </div>

            <!-- Plans Table -->
            <div class="card">
                <div class="card-body">
                    <h5 class="mb-3">All Plans</h5>

                    <?php if (empty($plans)): ?>
                        <div class="alert alert-info">
                            <strong>ℹ️ No plans found!</strong> Click "New Plan" to create your first subscription plan.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Plan Name</th>
                                        <th>Duration</th>
                                        <th>Search Limit</th>
                                        <th>Price</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($plans as $plan): ?>
                                        <tr>
                                            <td><strong><?php echo htmlspecialchars($plan['plan_name']); ?></strong></td>
                                            <td><?php echo $plan['duration_days']; ?> days</td>
                                            <td><?php echo $plan['search_limit']; ?></td>
                                            <td><strong>₹<?php echo number_format($plan['price'], 2); ?></strong></td>
                                            <td>
                                                <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#editModal<?php echo $plan['id']; ?>">Edit</button>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="action" value="delete_plan">
                                                    <input type="hidden" name="plan_id" value="<?php echo $plan['id']; ?>">
                                                    <button type="submit" class="btn btn-sm btn-outline-danger" onclick="return confirm('Delete this plan?');">Delete</button>
                                                </form>
                                            </td>
                                        </tr>

                                        <!-- Edit Modal -->
                                        <div class="modal fade" id="editModal<?php echo $plan['id']; ?>" tabindex="-1">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title">Edit Plan</h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                                    </div>
                                                    <form method="POST">
                                                        <div class="modal-body">
                                                            <input type="hidden" name="action" value="update_plan">
                                                            <input type="hidden" name="plan_id" value="<?php echo $plan['id']; ?>">

                                                            <div class="mb-3">
                                                                <label class="form-label">Plan Name</label>
                                                                <input type="text" name="plan_name" class="form-control" value="<?php echo htmlspecialchars($plan['plan_name']); ?>" required>
                                                            </div>

                                                            <div class="row">
                                                                <div class="col-md-6 mb-3">
                                                                    <label class="form-label">Duration (Days)</label>
                                                                    <input type="number" name="duration_days" class="form-control" value="<?php echo $plan['duration_days']; ?>" required>
                                                                </div>
                                                                <div class="col-md-6 mb-3">
                                                                    <label class="form-label">Search Limit</label>
                                                                    <input type="number" name="search_limit" class="form-control" value="<?php echo $plan['search_limit']; ?>" required>
                                                                </div>
                                                            </div>

                                                            <div class="mb-3">
                                                                <label class="form-label">Price (₹)</label>
                                                                <input type="number" name="price" class="form-control" value="<?php echo $plan['price']; ?>" step="0.01" required>
                                                            </div>

                                                            <div class="mb-3">
                                                                <label class="form-label">Description</label>
                                                                <textarea name="description" class="form-control" rows="2"><?php echo htmlspecialchars($plan['description']); ?></textarea>
                                                            </div>

                                                            <div class="mb-3">
                                                                <label class="form-label">Card Color 🎨</label>
                                                                <div style="display: flex; gap: 10px; align-items: center;">
                                                                    <input type="color" name="card_color" class="form-control cardColorInput" id="cardColorInput<?php echo $plan['id']; ?>" value="<?php echo htmlspecialchars($plan['card_color'] ?? '#007AFF'); ?>" style="width: 80px; height: 40px; border: 2px solid #ddd; cursor: pointer;">
                                                                    <div style="display: flex; gap: 8px;">
                                                                        <button type="button" class="btn btn-sm" style="background: #FF9500; border: none; color: white;" onclick="document.getElementById('cardColorInput<?php echo $plan['id']; ?>').value='#FF9500'">Orange</button>
                                                                        <button type="button" class="btn btn-sm" style="background: #FF3B30; border: none; color: white;" onclick="document.getElementById('cardColorInput<?php echo $plan['id']; ?>').value='#FF3B30'">Red</button>
                                                                        <button type="button" class="btn btn-sm" style="background: #34C759; border: none; color: white;" onclick="document.getElementById('cardColorInput<?php echo $plan['id']; ?>').value='#34C759'">Green</button>
                                                                        <button type="button" class="btn btn-sm" style="background: #007AFF; border: none; color: white;" onclick="document.getElementById('cardColorInput<?php echo $plan['id']; ?>').value='#007AFF'">Blue</button>
                                                                    </div>
                                                                </div>
                                                            </div>

                                                            <div class="form-check">
                                                                <input type="checkbox" name="is_active" class="form-check-input" id="active<?php echo $plan['id']; ?>" <?php echo $plan['is_active'] ? 'checked' : ''; ?>>
                                                                <label class="form-check-label" for="active<?php echo $plan['id']; ?>">Active</label>
                                                            </div>
                                                        </div>
                                                        <div class="modal-footer">
                                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                            <button type="submit" class="btn btn-primary">Update Plan</button>
                                                        </div>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Create Plan Modal -->
    <div class="modal fade" id="createPlanModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Create New Plan</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="create_plan">

                        <div class="mb-3">
                            <label class="form-label">Plan Name *</label>
                            <input type="text" name="plan_name" class="form-control" placeholder="e.g., Basic Plan" required>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Duration (Days) *</label>
                                <input type="number" name="duration_days" class="form-control" placeholder="30" min="1" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Search Limit *</label>
                                <input type="number" name="search_limit" class="form-control" placeholder="100" min="1" required>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Price (₹) *</label>
                            <input type="number" name="price" class="form-control" placeholder="99.99" min="0.01" step="0.01" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Description</label>
                            <textarea name="description" class="form-control" rows="3" placeholder="Plan description..."></textarea>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Card Color 🎨</label>
                            <div style="display: flex; gap: 10px; align-items: center;">
                                <input type="color" name="card_color" class="form-control" id="cardColorInput" value="#007AFF" style="width: 80px; height: 40px; border: 2px solid #ddd; cursor: pointer;">
                                <div style="display: flex; gap: 8px;">
                                    <button type="button" class="btn btn-sm" style="background: #FF9500; border: none; color: white;" onclick="document.getElementById('cardColorInput').value='#FF9500'">Orange</button>
                                    <button type="button" class="btn btn-sm" style="background: #FF3B30; border: none; color: white;" onclick="document.getElementById('cardColorInput').value='#FF3B30'">Red</button>
                                    <button type="button" class="btn btn-sm" style="background: #34C759; border: none; color: white;" onclick="document.getElementById('cardColorInput').value='#34C759'">Green</button>
                                    <button type="button" class="btn btn-sm" style="background: #007AFF; border: none; color: white;" onclick="document.getElementById('cardColorInput').value='#007AFF'">Blue</button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Create Plan</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        feather.replace();
    </script>
</body>
</html>
