<?php
/**
 * Face Recognition API
 * Handles face scanning, registration, and authentication
 */

header('Content-Type: application/json');
error_reporting(E_ALL);
ini_set('display_errors', 0);

require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';
require_once '../classes/FaceRecognition.php';

$db = Database::getInstance();
$auth = new Auth($db);
$faceRecognition = new FaceRecognition($db);

$action = $_POST['action'] ?? $_GET['action'] ?? null;
$response = ['success' => false, 'message' => ''];

try {
    switch ($action) {
        case 'register_face':
            // Register face for logged-in user
            if (!$auth->isLoggedIn()) {
                throw new Exception('Not authenticated');
            }
            
            $faceData = $_POST['face_data'] ?? null;
            if (!$faceData) {
                throw new Exception('No face data provided');
            }
            
            $currentUser = $auth->getCurrentUser();
            if (!$currentUser || !isset($currentUser['id'])) {
                throw new Exception('User ID not found');
            }
            
            // Log for debugging
            error_log('Face registration attempt for user: ' . $currentUser['id'] . ', data length: ' . strlen($faceData));
            
            if ($faceRecognition->saveFaceData($currentUser['id'], $faceData)) {
                $response['success'] = true;
                $response['message'] = 'Face registered successfully';
                $response['user_id'] = $currentUser['id'];
                error_log('Face registration successful for user: ' . $currentUser['id']);
            } else {
                throw new Exception('Failed to save face data');
            }
            break;
        
        case 'check_unregistered_faces':
            // Check if any faces are registered in the system
            $users = $db->query(
                'SELECT COUNT(*) as count FROM users WHERE face_data IS NOT NULL AND face_data != ""'
            )->fetch(PDO::FETCH_ASSOC);
            
            $response['success'] = true;
            $response['faces_available'] = $users['count'] ?? 0;
            break;
        
        case 'authenticate_face':
            // Authenticate user by face
            $faceData = $_POST['face_data'] ?? null;
            if (!$faceData) {
                throw new Exception('No face data provided');
            }
            
            $user = $faceRecognition->getUserByFace($faceData);
            if (!$user) {
                throw new Exception('Face not recognized');
            }
            
            // Set session
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_name'] = $user['name'];
            $_SESSION['login_time'] = time();
            $_SESSION['login_type'] = 'face_id';
            
            $response['success'] = true;
            $response['message'] = 'Face authentication successful';
            $response['user_id'] = $user['id'];
            
            // Determine redirect based on user role
            $userRole = $db->fetch('SELECT role FROM users WHERE id = ?', [$user['id']]);
            if ($userRole && $userRole['role'] === 'admin') {
                $response['redirect'] = 'admin/dashboard.php';
            } else {
                $response['redirect'] = 'user/dashboard.php';
            }
            break;
        
        case 'check_face_enabled':
            // Check if user has face registration
            if (!$auth->isLoggedIn()) {
                throw new Exception('Not authenticated');
            }
            
            $currentUser = $auth->getCurrentUser();
            $hasFace = $faceRecognition->hasFaceData($currentUser['id']);
            
            $response['success'] = true;
            $response['has_face'] = $hasFace;
            $response['face_enabled'] = $hasFace;
            break;
        
        case 'delete_face':
            // Delete face registration
            if (!$auth->isLoggedIn()) {
                throw new Exception('Not authenticated');
            }
            
            $currentUser = $auth->getCurrentUser();
            if ($faceRecognition->deleteFaceData($currentUser['id'])) {
                $response['success'] = true;
                $response['message'] = 'Face data deleted successfully';
            } else {
                throw new Exception('Failed to delete face data');
            }
            break;
        
        default:
            throw new Exception('Invalid action');
    }
} catch (Exception $e) {
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    $response['error'] = $e->getMessage();
}

echo json_encode($response);
?>
