<?php
// Analytics API - Get user's search statistics
header('Content-Type: application/json');
require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';

$db = Database::getInstance();
$auth = new Auth($db);

if (!$auth->isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Not logged in']);
    exit();
}

$userId = $auth->getCurrentUser()['id'];

// Total searches
$totalResult = $db->fetch(
    "SELECT COUNT(*) as count FROM search_logs WHERE user_id = ?",
    [$userId]
);
$totalSearches = $totalResult['count'] ?? 0;

// Mobile searches
$mobileResult = $db->fetch(
    "SELECT COUNT(*) as count FROM search_logs WHERE user_id = ? AND type = 'mobile'",
    [$userId]
);
$mobileSearches = $mobileResult['count'] ?? 0;

// Aadhaar searches
$aadhaarResult = $db->fetch(
    "SELECT COUNT(*) as count FROM search_logs WHERE user_id = ? AND type = 'aadhaar'",
    [$userId]
);
$aadhaarSearches = $aadhaarResult['count'] ?? 0;

// Credits spent
$creditsResult = $db->fetch(
    "SELECT SUM(cost) as total FROM search_logs WHERE user_id = ? AND status = 'success'",
    [$userId]
);
$creditsSpent = $creditsResult['total'] ?? 0;

// Current subscription and balance
require_once '../classes/SubscriptionManager.php';
$subMgr = new SubscriptionManager();
$userSubscription = $subMgr->getActiveSubscription($userId);
$userTrial = $subMgr->getFreeTrial($userId);

// Get remaining searches from active subscription
$searchesRemaining = 0;
if ($userSubscription && $userSubscription['status'] === 'active') {
    $searchesRemaining = (int)$userSubscription['searches_remaining'];
} elseif ($userTrial && $userTrial['is_trial_active']) {
    $searchesRemaining = (int)$userTrial['searches_remaining'];
}

// Wallet balance (for legacy)
$balanceResult = $db->fetch(
    "SELECT balance FROM wallets WHERE user_id = ?",
    [$userId]
);
$currentBalance = $balanceResult['balance'] ?? 0;

// Success rate
$successResult = $db->fetch(
    "SELECT COUNT(*) as count FROM search_logs WHERE user_id = ? AND status = 'success'",
    [$userId]
);
$successCount = $successResult['count'] ?? 0;
$successRate = $totalSearches > 0 ? round(($successCount / $totalSearches) * 100) : 0;

// Check if user is trial - show loading state for trial users
$isTrialUser = $userTrial && $userTrial['is_trial_active'] && !$userSubscription;

if ($isTrialUser) {
    // For trial users, return loading state
    echo json_encode([
        'success' => true,
        'is_loading' => true,
        'total_searches' => 0,
        'mobile_searches' => 0,
        'aadhaar_searches' => 0,
        'searches_remaining' => (int)$userTrial['searches_remaining'],
        'success_rate' => 0,
        'subscription_status' => 'trial',
        'subscription_name' => 'Free Trial',
        'days_remaining' => 0,
        'recent_searches' => []
    ]);
    exit();
}

// Recent searches (limit 10)
$recentSearches = $db->fetchAll(
    "SELECT type, query, created_at FROM search_logs WHERE user_id = ? ORDER BY created_at DESC LIMIT 10",
    [$userId]
);

// Format recent searches with time ago
$formattedRecent = [];
foreach ($recentSearches as $search) {
    $time = strtotime($search['created_at']);
    $now = time();
    $diff = $now - $time;
    
    if ($diff < 60) {
        $timeAgo = 'just now';
    } elseif ($diff < 3600) {
        $timeAgo = floor($diff / 60) . ' min ago';
    } elseif ($diff < 86400) {
        $timeAgo = floor($diff / 3600) . ' h ago';
    } else {
        $timeAgo = floor($diff / 86400) . ' d ago';
    }
    
    $formattedRecent[] = [
        'type' => $search['type'],
        'query' => htmlspecialchars($search['query']),
        'time_ago' => $timeAgo
    ];
}

echo json_encode([
    'success' => true,
    'is_loading' => false,
    'total_searches' => $totalSearches,
    'mobile_searches' => $mobileSearches,
    'aadhaar_searches' => $aadhaarSearches,
    'searches_remaining' => $searchesRemaining,
    'success_rate' => $successRate,
    'subscription_status' => $userSubscription ? 'premium' : 'free',
    'subscription_name' => $userSubscription ? $userSubscription['plan_name'] : 'No Active Plan',
    'days_remaining' => $userSubscription ? $userSubscription['days_remaining'] : 0,
    'recent_searches' => $formattedRecent
]);
?>
