<?php
// api/razorpay_callback.php - Webhook endpoint for Razorpay payment confirmations
header('Content-Type: application/json');
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/RazorpayPayment.php';

$db = Database::getInstance();

// Razorpay webhook verification
$data = json_decode(file_get_contents('php://input'), true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid request']);
    exit;
}

// Log incoming webhook
error_log('Razorpay Webhook: ' . json_encode($data));

// Handle different webhook events
$event = $data['event'] ?? '';

switch ($event) {
    case 'payment.authorized':
        handlePaymentAuthorized($data['payload']['payment']['entity'], $db);
        break;
    
    case 'payment.failed':
        handlePaymentFailed($data['payload']['payment']['entity'], $db);
        break;
    
    case 'payment.captured':
        handlePaymentCaptured($data['payload']['payment']['entity'], $db);
        break;
    
    default:
        error_log('Unknown event: ' . $event);
}

http_response_code(200);
echo json_encode(['status' => 'received']);

/**
 * Handle successful payment capture
 */
function handlePaymentCaptured($paymentData, $db) {
    $paymentId = $paymentData['id'];
    $orderId = $paymentData['order_id'];
    $amount = $paymentData['amount'] / 100; // Convert paise to rupees
    
    // Find subscription by Razorpay order ID
    $subscription = $db->fetch(
        'SELECT * FROM user_subscriptions WHERE razorpay_order_id = ?',
        [$orderId]
    );
    
    if (!$subscription) {
        error_log("Subscription not found for order: $orderId");
        return;
    }
    
    // Update subscription status
    $db->update('user_subscriptions', [
        'razorpay_payment_id' => $paymentId,
        'payment_status' => 'success',
        'status' => 'active'
    ], 'id', $subscription['id']);
    
    // Mark trial as ended if exists
    $trial = $db->fetch('SELECT * FROM free_trial_tracking WHERE user_id = ?', [$subscription['user_id']]);
    if ($trial && $trial['is_trial_active']) {
        $db->query('UPDATE free_trial_tracking SET is_trial_active = false, trial_ended_at = NOW() WHERE user_id = ?', 
            [$subscription['user_id']]);
    }
    
    // Log history
    $db->insert('subscription_history', [
        'user_id' => $subscription['user_id'],
        'subscription_id' => $subscription['id'],
        'action' => 'payment_captured',
        'old_status' => 'pending',
        'new_status' => 'active',
        'details' => json_encode([
            'razorpay_payment_id' => $paymentId,
            'amount' => $amount,
            'timestamp' => date('Y-m-d H:i:s')
        ])
    ]);
    
    error_log("Payment captured successfully for subscription: {$subscription['id']}");
}

/**
 * Handle payment authorization
 */
function handlePaymentAuthorized($paymentData, $db) {
    $paymentId = $paymentData['id'];
    $orderId = $paymentData['order_id'];
    
    // Find subscription
    $subscription = $db->fetch(
        'SELECT * FROM user_subscriptions WHERE razorpay_order_id = ?',
        [$orderId]
    );
    
    if ($subscription) {
        $db->update('user_subscriptions', [
            'razorpay_payment_id' => $paymentId,
            'payment_status' => 'success'
        ], 'id', $subscription['id']);
        
        error_log("Payment authorized for subscription: {$subscription['id']}");
    }
}

/**
 * Handle payment failure
 */
function handlePaymentFailed($paymentData, $db) {
    $paymentId = $paymentData['id'];
    $orderId = $paymentData['order_id'];
    $errorCode = $paymentData['error_code'] ?? 'unknown';
    $errorDescription = $paymentData['error_description'] ?? 'Payment failed';
    
    // Find subscription
    $subscription = $db->fetch(
        'SELECT * FROM user_subscriptions WHERE razorpay_order_id = ?',
        [$orderId]
    );
    
    if ($subscription) {
        $db->update('user_subscriptions', [
            'razorpay_payment_id' => $paymentId,
            'payment_status' => 'failed',
            'status' => 'expired'
        ], 'id', $subscription['id']);
        
        // Log history
        $db->insert('subscription_history', [
            'user_id' => $subscription['user_id'],
            'subscription_id' => $subscription['id'],
            'action' => 'payment_failed',
            'new_status' => 'expired',
            'details' => json_encode([
                'error_code' => $errorCode,
                'error_description' => $errorDescription,
                'timestamp' => date('Y-m-d H:i:s')
            ])
        ]);
        
        error_log("Payment failed for subscription {$subscription['id']}: $errorDescription");
    }
}
?>
