<?php
// api/razorpay_payment.php

header('Content-Type: application/json');

session_start();
require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/SubscriptionManager.php';
require_once '../classes/RazorpayPayment.php';

$db = Database::getInstance();
$subscriptionMgr = new SubscriptionManager();
$razorpay = new RazorpayPayment(RAZORPAY_KEY_ID, RAZORPAY_KEY_SECRET);

$action = $_GET['action'] ?? $_POST['action'] ?? '';

switch ($action) {
    case 'verify_payment':
        verifyPayment();
        break;
    case 'webhook':
        handleWebhook();
        break;
    case 'get_order_details':
        getOrderDetails();
        break;
    default:
        http_response_code(400);
        echo json_encode(['status' => false, 'message' => 'Invalid action']);
}

/**
 * Verify payment after user completes Razorpay checkout
 */
function verifyPayment() {
    global $razorpay, $db, $subscriptionMgr;
    
    if (!isset($_SESSION['user_id'])) {
        http_response_code(401);
        echo json_encode(['status' => false, 'message' => 'Unauthorized']);
        return;
    }
    
    $userId = $_SESSION['user_id'];
    $orderId = $_POST['razorpay_order_id'] ?? null;
    $paymentId = $_POST['razorpay_payment_id'] ?? null;
    $signature = $_POST['razorpay_signature'] ?? null;
    
    if (!$orderId || !$paymentId || !$signature) {
        echo json_encode(['status' => false, 'message' => 'Missing payment details']);
        return;
    }
    
    // Verify signature
    $verifyResult = $razorpay->verifyPaymentSignature($orderId, $paymentId, $signature);
    
    if (!$verifyResult['status']) {
        echo json_encode(['status' => false, 'message' => 'Payment verification failed']);
        return;
    }
    
    // Get subscription details from session or database
    $subscription = $db->fetch(
        'SELECT id FROM user_subscriptions WHERE razorpay_order_id = ? AND user_id = ?',
        [$orderId, $userId]
    );
    
    if (!$subscription) {
        echo json_encode(['status' => false, 'message' => 'Subscription not found']);
        return;
    }
    
    // Confirm payment in database
    $confirmResult = $subscriptionMgr->confirmPayment($userId, $subscription['id'], $paymentId, $signature);
    
    if ($confirmResult['status']) {
        // Log transaction
        $db->insert('subscription_history', [
            'user_id' => $userId,
            'subscription_id' => $subscription['id'],
            'action' => 'payment_verified',
            'new_status' => 'active',
            'details' => json_encode(['payment_id' => $paymentId])
        ]);
        
        echo json_encode([
            'status' => true,
            'message' => 'Payment verified successfully',
            'subscription_id' => $subscription['id']
        ]);
    } else {
        echo json_encode(['status' => false, 'message' => 'Failed to confirm payment']);
    }
}

/**
 * Handle Razorpay webhook
 */
function handleWebhook() {
    global $razorpay, $db, $subscriptionMgr;
    
    $input = file_get_contents('php://input');
    $webhookSignature = $_SERVER['HTTP_X_RAZORPAY_SIGNATURE'] ?? null;
    
    if (!$webhookSignature) {
        http_response_code(400);
        echo json_encode(['status' => false, 'message' => 'Missing signature']);
        return;
    }
    
    // Verify webhook signature
    if (!$razorpay->verifyWebhookSignature($input, $webhookSignature)) {
        http_response_code(401);
        echo json_encode(['status' => false, 'message' => 'Invalid signature']);
        return;
    }
    
    $event = json_decode($input, true);
    $eventType = $event['event'] ?? null;
    
    switch ($eventType) {
        case 'payment.authorized':
            handlePaymentAuthorized($event);
            break;
        case 'payment.failed':
            handlePaymentFailed($event);
            break;
        case 'payment.captured':
            handlePaymentCaptured($event);
            break;
        default:
            // Unknown event
            error_log('Unknown webhook event: ' . $eventType);
    }
    
    echo json_encode(['status' => true]);
}

/**
 * Handle payment authorized webhook
 */
function handlePaymentAuthorized($event) {
    global $db, $subscriptionMgr;
    
    $paymentData = $event['payload']['payment']['entity'] ?? null;
    if (!$paymentData) return;
    
    $paymentId = $paymentData['id'];
    $orderId = $paymentData['order_id'];
    
    // Find subscription by order ID
    $subscription = $db->fetch(
        'SELECT user_id FROM user_subscriptions WHERE razorpay_order_id = ?',
        [$orderId]
    );
    
    if (!$subscription) {
        error_log('Subscription not found for order: ' . $orderId);
        return;
    }
    
    // Update payment status
    $db->query(
        'UPDATE user_subscriptions SET razorpay_payment_id = ?, payment_status = "success" WHERE razorpay_order_id = ?',
        [$paymentId, $orderId]
    );
}

/**
 * Handle payment failed webhook
 */
function handlePaymentFailed($event) {
    global $db;
    
    $paymentData = $event['payload']['payment']['entity'] ?? null;
    if (!$paymentData) return;
    
    $paymentId = $paymentData['id'];
    $orderId = $paymentData['order_id'];
    
    // Find subscription by order ID
    $subscription = $db->fetch(
        'SELECT user_id FROM user_subscriptions WHERE razorpay_order_id = ?',
        [$orderId]
    );
    
    if (!$subscription) return;
    
    // Update payment status
    $db->query(
        'UPDATE user_subscriptions SET payment_status = "failed", status = "cancelled" WHERE razorpay_order_id = ?',
        [$orderId]
    );
    
    // Log failed payment
    $db->insert('subscription_history', [
        'user_id' => $subscription['user_id'],
        'subscription_id' => $subscription['id'],
        'action' => 'payment_failed',
        'new_status' => 'cancelled',
        'details' => json_encode(['payment_id' => $paymentId])
    ]);
}

/**
 * Handle payment captured webhook
 */
function handlePaymentCaptured($event) {
    global $db;
    
    $paymentData = $event['payload']['payment']['entity'] ?? null;
    if (!$paymentData) return;
    
    $paymentId = $paymentData['id'];
    $orderId = $paymentData['order_id'];
    
    // Update subscription as confirmed
    $db->query(
        'UPDATE user_subscriptions SET razorpay_payment_id = ?, payment_status = "success", status = "active" WHERE razorpay_order_id = ?',
        [$paymentId, $orderId]
    );
}

/**
 * Get order details for checkout
 */
function getOrderDetails() {
    if (!isset($_SESSION['razorpay_order'])) {
        http_response_code(400);
        echo json_encode(['status' => false, 'message' => 'No order found']);
        return;
    }
    
    $order = $_SESSION['razorpay_order'];
    
    echo json_encode([
        'status' => true,
        'order_id' => $order['order_id'],
        'amount' => $order['amount'] * 100, // Convert to paise
        'key_id' => $order['key_id'],
        'subscription_id' => $order['subscription_id'],
        'plan_id' => $order['plan_id']
    ]);
}
?>
