<?php
/**
 * Verify OTP from Modal Popup
 * Used in user profile email verification
 */

header('Content-Type: application/json');

require_once '../config/config.php';
require_once '../classes/Database.php';
require_once '../classes/Auth.php';

$db = Database::getInstance();
$auth = new Auth($db);

// Check if user is logged in
if (!$auth->isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit();
}

$data = json_decode(file_get_contents('php://input'), true);
$otp = trim($data['otp'] ?? '');
$email = trim($data['email'] ?? '');

// Validate
if (empty($otp) || empty($email)) {
    echo json_encode(['success' => false, 'message' => 'Missing OTP or email']);
    exit();
}

if (strlen($otp) !== 6 || !ctype_digit($otp)) {
    echo json_encode(['success' => false, 'message' => 'OTP must be 6 digits']);
    exit();
}

// Get OTP record
$otpRecord = $db->fetch(
    'SELECT id, otp, attempts, expires_at FROM email_otps WHERE email = ? ORDER BY created_at DESC LIMIT 1',
    [$email]
);

if (!$otpRecord) {
    echo json_encode(['success' => false, 'message' => 'No OTP found. Please request a new one.']);
    exit();
}

// Check expiry using database time comparison
$expiryCheck = $db->fetch(
    'SELECT CASE WHEN NOW() > ? THEN "EXPIRED" ELSE "VALID" END as status',
    [$otpRecord['expires_at']]
);

if ($expiryCheck['status'] === 'EXPIRED') {
    echo json_encode(['success' => false, 'message' => 'OTP has expired. Please request a new one.']);
    exit();
}

if ($otpRecord['attempts'] >= 5) {
    echo json_encode(['success' => false, 'message' => 'Maximum attempts exceeded. Please request a new OTP.']);
    exit();
}

if ($otpRecord['otp'] === $otp) {
    // OTP is correct - Mark email as verified
    $db->query(
        'UPDATE users SET email_verified = 1, email_verified_at = NOW() WHERE email = ?',
        [$email]
    );
    
    // Delete used OTP
    $db->query(
        'DELETE FROM email_otps WHERE email = ?',
        [$email]
    );
    
    echo json_encode(['success' => true, 'message' => 'Email verified successfully!']);
    exit();
} else {
    // Wrong OTP - increment attempts
    $db->update(
        'email_otps',
        ['attempts' => $otpRecord['attempts'] + 1],
        'id = ?',
        [$otpRecord['id']]
    );
    
    $remaining = 5 - ($otpRecord['attempts'] + 1);
    if ($remaining > 0) {
        echo json_encode(['success' => false, 'message' => "Invalid OTP. $remaining attempts remaining."]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Maximum attempts exceeded. Please request a new OTP.']);
    }
    exit();
}
