<?php
class Database {
    private $host = 'localhost';
    private $db_name = 'iplztpse_final';
    private $username = 'root';
    private $password = '';
    private static $instance = null;
    private $conn;

    private function __construct() {
        try {
            $this->conn = new PDO(
                'mysql:host=' . $this->host . ';dbname=' . $this->db_name . ';charset=utf8',
                $this->username,
                $this->password
            );
            $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->conn->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        } catch(PDOException $exception) {
            die('❌ Database Connection Error: ' . $exception->getMessage());
        }
    }

    // ✅ FIXED: return Database object, not raw PDO
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance; 
    }

    public function getConnection() {
        return $this->conn;
    }

    // Query wrapper
    public function query($sql, $params = []) {
        try {
            $stmt = $this->conn->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch(PDOException $e) {
            error_log('Database query error: ' . $e->getMessage());
            return false;
        }
    }

    // Fetch single record
    public function fetch($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt ? $stmt->fetch() : false;
    }

    // Fetch all records
    public function fetchAll($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt ? $stmt->fetchAll() : [];
    }

    // Insert record
    public function insert($table, $data) {
        try {
            $columns = array_keys($data);
            $placeholders = array_map(fn($col) => ':' . $col, $columns);

            $sql = "INSERT INTO {$table} (" . implode(', ', $columns) . 
                   ") VALUES (" . implode(', ', $placeholders) . ")";

            $stmt = $this->conn->prepare($sql);
            if (!$stmt) {
                error_log('Database insert error: Failed to prepare statement. SQL: ' . $sql);
                return false;
            }
            
            // Build params array with keys prefixed with :
            $params = [];
            foreach ($data as $key => $value) {
                $params[':' . $key] = $value;
            }
            
            $result = $stmt->execute($params);
            if (!$result) {
                error_log('Database insert error: Execute failed. SQL: ' . $sql . ' | Params: ' . json_encode($params) . ' | Error: ' . json_encode($stmt->errorInfo()));
                return false;
            }
            
            return $this->conn->lastInsertId();
        } catch(PDOException $e) {
            error_log('Database insert error: ' . $e->getMessage());
            return false;
        }
    }

    // Update record
    public function update($table, $data, $where, $whereParams = []) {
        $set = [];
        $allParams = [];
        
        // Build SET clause with column = value pairs
        foreach ($data as $key => $value) {
            $set[] = "{$key} = ?";
            $allParams[] = $value;
        }

        // Add WHERE parameters
        foreach ($whereParams as $value) {
            $allParams[] = $value;
        }

        $sql = "UPDATE {$table} SET " . implode(', ', $set) . " WHERE " . $where;
        
        try {
            $stmt = $this->conn->prepare($sql);
            if (!$stmt) {
                error_log('Database update error: Failed to prepare statement. SQL: ' . $sql);
                return false;
            }
            $result = $stmt->execute($allParams);
            if (!$result) {
                error_log('Database update error: Execute failed. SQL: ' . $sql . ' | Params: ' . json_encode($allParams) . ' | Error: ' . json_encode($stmt->errorInfo()));
                return false;
            }
            return $stmt->rowCount();
        } catch(PDOException $e) {
            error_log('Database update error: ' . $e->getMessage() . ' | SQL: ' . $sql);
            return false;
        }
    }

    // Delete record
    public function delete($table, $where, $params = []) {
        $sql = "DELETE FROM {$table} WHERE {$where}";
        $stmt = $this->query($sql, $params);
        return $stmt ? $stmt->rowCount() : false;
    }
}
?>
