<?php
/**
 * Face Recognition Handler
 * Handles face data storage, comparison, and authentication
 */

class FaceRecognition {
    private $db;
    private $tolerance = 0.6; // Face matching tolerance (0-1, lower is stricter)
    
    public function __construct($database) {
        $this->db = $database;
    }
    
    /**
     * Save face data for a user
     * @param int $userId User ID
     * @param string $faceData Base64 encoded face descriptor
     * @return bool Success status
     */
    public function saveFaceData($userId, $faceData) {
        try {
            // Validate face data
            if (empty($faceData) || strlen($faceData) < 100) {
                error_log('Face save failed: invalid data length ' . strlen($faceData));
                return false;
            }
            
            // Check if user exists
            $user = $this->db->fetch('SELECT id FROM users WHERE id = ?', [$userId]);
            if (!$user) {
                error_log('Face save failed: user not found ' . $userId);
                return false;
            }
            
            // Update face data
            $result = $this->db->update(
                'users',
                [
                    'face_data' => $faceData,
                    'face_last_updated' => date('Y-m-d H:i:s')
                ],
                'id = ?',
                [$userId]
            );
            
            // Result is either false on error, or row count (0 or 1) on success
            if ($result === false) {
                error_log('Face save failed: database update returned false');
                return false;
            }
            
            error_log('Face save successful: updated ' . $result . ' row(s)');
            return true; // Success (row count doesn't matter, update succeeded)
        } catch (Exception $e) {
            error_log('Face save error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get user by face data matching
     * @param string $faceData Base64 encoded face descriptor
     * @return array|null User data if match found
     */
    public function getUserByFace($faceData) {
        try {
            if (empty($faceData)) {
                return null;
            }
            
            // Get all users with face data
            $users = $this->db->query(
                'SELECT id, email, name, face_data FROM users WHERE face_data IS NOT NULL AND face_data != ""'
            )->fetchAll(PDO::FETCH_ASSOC);
            
            if (empty($users)) {
                return null;
            }
            
            $incomingDescriptor = json_decode($faceData, true);
            if (!is_array($incomingDescriptor)) {
                return null;
            }
            
            // Compare with each user's face
            foreach ($users as $user) {
                $storedDescriptor = json_decode($user['face_data'], true);
                if (!is_array($storedDescriptor)) {
                    continue;
                }
                
                $distance = $this->calculateDistance($incomingDescriptor, $storedDescriptor);
                
                if ($distance <= $this->tolerance) {
                    return $user;
                }
            }
            
            return null;
        } catch (Exception $e) {
            error_log('Face recognition error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Calculate Euclidean distance between two face descriptors
     * @param array $descriptor1 First descriptor
     * @param array $descriptor2 Second descriptor
     * @return float Distance value
     */
    private function calculateDistance($descriptor1, $descriptor2) {
        if (count($descriptor1) !== count($descriptor2)) {
            return PHP_FLOAT_MAX;
        }
        
        $sum = 0;
        for ($i = 0; $i < count($descriptor1); $i++) {
            $diff = (float)$descriptor1[$i] - (float)$descriptor2[$i];
            $sum += $diff * $diff;
        }
        
        return sqrt($sum);
    }
    
    /**
     * Check if user has face data registered
     * @param int $userId User ID
     * @return bool
     */
    public function hasFaceData($userId) {
        try {
            $user = $this->db->fetch(
                'SELECT face_data FROM users WHERE id = ? AND face_data IS NOT NULL AND face_data != ""',
                [$userId]
            );
            return $user !== null;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Delete face data for a user
     * @param int $userId User ID
     * @return bool
     */
    public function deleteFaceData($userId) {
        try {
            $this->db->update(
                'users',
                ['face_data' => null, 'face_last_updated' => null],
                'id = ?',
                [$userId]
            );
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Set face matching tolerance (0-1)
     * @param float $tolerance Lower = stricter matching
     */
    public function setTolerance($tolerance) {
        if ($tolerance >= 0 && $tolerance <= 1) {
            $this->tolerance = $tolerance;
        }
    }
}
?>
