<?php
// classes/RazorpayPayment.php

class RazorpayPayment {
    private $keyId;
    private $keySecret;
    private $db;
    
    public function __construct($keyId, $keySecret) {
        $this->keyId = $keyId;
        $this->keySecret = $keySecret;
        $this->db = Database::getInstance();
    }
    
    /**
     * Create Razorpay order for subscription
     */
    public function createOrder($userId, $planId, $subscriptionId) {
        try {
            $plan = $this->db->fetch(
                'SELECT * FROM subscription_plans WHERE id = ?',
                [$planId]
            );
            
            if (!$plan) {
                return ['status' => false, 'message' => 'Plan not found'];
            }
            
            // Convert rupees to paise (Razorpay uses paise)
            $amountInPaise = $plan['price'] * 100;
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'https://api.razorpay.com/v1/orders');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_USERPWD, $this->keyId . ':' . $this->keySecret);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
                'amount' => $amountInPaise,
                'currency' => 'INR',
                'receipt' => 'SUB_' . $subscriptionId . '_' . time(),
                'notes' => [
                    'user_id' => $userId,
                    'subscription_id' => $subscriptionId,
                    'plan_id' => $planId,
                    'plan_name' => $plan['plan_name']
                ]
            ]));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                $orderData = json_decode($response, true);
                return [
                    'status' => true,
                    'order_id' => $orderData['id'],
                    'amount' => $plan['price'],
                    'key_id' => $this->keyId,
                    'plan_name' => $plan['plan_name']
                ];
            } else {
                error_log('Razorpay Order Creation Error: ' . $response);
                return ['status' => false, 'message' => 'Failed to create order'];
            }
        } catch (Exception $e) {
            error_log('Create Order Exception: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Error creating order'];
        }
    }
    
    /**
     * Verify payment signature
     */
    public function verifyPaymentSignature($orderId, $paymentId, $signature) {
        try {
            $signaturePayload = $orderId . '|' . $paymentId;
            $expectedSignature = hash_hmac('sha256', $signaturePayload, $this->keySecret);
            
            if ($expectedSignature === $signature) {
                return ['status' => true, 'verified' => true];
            } else {
                return ['status' => false, 'message' => 'Invalid signature'];
            }
        } catch (Exception $e) {
            error_log('Verify Signature Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Error verifying signature'];
        }
    }
    
    /**
     * Get payment details from Razorpay
     */
    public function getPaymentDetails($paymentId) {
        try {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'https://api.razorpay.com/v1/payments/' . $paymentId);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_USERPWD, $this->keyId . ':' . $this->keySecret);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                return [
                    'status' => true,
                    'data' => json_decode($response, true)
                ];
            } else {
                return ['status' => false, 'message' => 'Failed to fetch payment details'];
            }
        } catch (Exception $e) {
            error_log('Get Payment Details Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Error fetching payment details'];
        }
    }
    
    /**
     * Handle webhook verification
     */
    public function verifyWebhookSignature($webhookBody, $webhookSignature) {
        try {
            $expectedSignature = hash_hmac('sha256', $webhookBody, $this->keySecret);
            
            if ($expectedSignature === $webhookSignature) {
                return true;
            }
            return false;
        } catch (Exception $e) {
            error_log('Webhook Verification Error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Refund payment
     */
    public function refundPayment($paymentId, $amount = null) {
        try {
            $url = 'https://api.razorpay.com/v1/payments/' . $paymentId . '/refund';
            
            $postData = [];
            if ($amount) {
                $postData['amount'] = $amount * 100; // Convert to paise
            }
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_USERPWD, $this->keyId . ':' . $this->keySecret);
            
            if (!empty($postData)) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
            }
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                $refundData = json_decode($response, true);
                return [
                    'status' => true,
                    'refund_id' => $refundData['id'],
                    'amount' => $refundData['amount'] / 100
                ];
            } else {
                return ['status' => false, 'message' => 'Failed to process refund'];
            }
        } catch (Exception $e) {
            error_log('Refund Payment Error: ' . $e->getMessage());
            return ['status' => false, 'message' => 'Error processing refund'];
        }
    }
}
?>
